#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright Nicolai Bæk Thomsen
# Author: Nicolai Bæk Thomsen, Dept. Electronic Systems, Aalborg University
# Date: 3/3 2017
# 
################################################################################
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You can obtain a copy of the GNU General Public License from
#   http://www.gnu.org/copyleft/gpl.html or by writing to
#   Free Software Foundation, Inc.,675 Mass Ave, Cambridge, MA 02139, USA.
################################################################################
import sys, time
import os
import glob
import subprocess
import shutil
from subprocess import call
from subprocess import check_output
import roslib; 
import rospy
import pickle
import numpy as np
import matplotlib.pyplot as plt
from scipy import signal
import scipy.io.wavfile as sc_wav
import scipy.stats as sc_stat
from std_msgs.msg import Float32
from navigation.msg import RelativeMove
import numpy.linalg as LA
from move_base_msgs.msg import *
from geometry_msgs.msg import *
from actionlib_msgs.msg import *
from actionlib.msg import *
from std_msgs.msg import Float32
import tf
from sound_play.libsoundplay import SoundClient
import wave
import contextlib
from speech_navigation.srv import *
from tf.transformations import euler_from_quaternion
from nav_msgs.msg import Odometry
from nav_msgs.msg import OccupancyGrid
from sensor_msgs.msg import LaserScan
from kobuki_msgs.msg import BumperEvent
from scipy.stats import multivariate_normal

M_SPACE = 0.226 #Spacing between most outer microphones
C = 342 #Speed of sound
E_THRES = 1e-8
E_THRES1 = 1e-8
FEATURE_THRESHOLD = 0.65
GAP_SIZE_THRES = 0.15 #Seems to work well with 0.15


DRR_INTERVAL = [0.3,4]
ENERGY_INTERVAL = [0.,22.]



def map_angle2front(angle):
    return angle - np.sign(angle)*180

def normalize_features(features):
    #Normalize features according to x_norm = (x-x_min) / (x_max - x_min)
    #Assumes that energy is first and then drr is next
    norm_features = np.zeros((features.shape))
    norm_features[0] = (features[0]-ENERGY_INTERVAL[0]) / (ENERGY_INTERVAL[1] - ENERGY_INTERVAL[0])
    norm_features[1] = (features[1]-DRR_INTERVAL[0]) / (DRR_INTERVAL[1] - DRR_INTERVAL[0]) 
    norm_features[norm_features<0.01] = 0.01
    norm_features[norm_features>0.99] = 0.99

    return norm_features


def map_angle2gap(angle,gap_start,gap_size):
    # This returns the gap which "matches" the doa from audio. It first checks if doa is in a visibility gap, if not,
    # then it finds the one with the closest edge
    N_gaps = len(gap_start)

    #Check if angle is within a gap
    gap_idx = -1
    for ii in range(N_gaps):
        if angle>gap_start[ii] and angle < (gap_start[ii]+gap_size[ii]):
            gap_idx = ii
            break

    #If not within a gap, then find the one with the boundary to the doa
    if gap_idx == -1:
        gap_stop = gap_start+gap_size
        gap_interval = np.vstack((gap_start,gap_stop)).T
        #print gap_interval.shape, gap_interval
        angle_diff = np.abs(gap_interval-angle)
        min_idx = np.where(angle_diff == np.min(angle_diff))
        gap_idx = min_idx[0]


    return gap_idx


def combine_doa(doa_front,doa_side):
    if doa_side == 1:
        fus_doa = doa_front
    else:
        if doa_front <= 0.: 
            fus_doa = -180+np.abs(doa_front)
        else:
            fus_doa = 180-doa_front#(np.abs(doa_side)+90. + 180-doa_front)/2.

    return fus_doa

def nan_helper(y):
    return np.isnan(y), lambda z: z.nonzero()[0]


class findSoundSource:
    def __init__(self):
        self.DEV_FRONT = rospy.get_param("~DEV_FRONT")
        self.DEV_BACK = rospy.get_param("~DEV_BACK")
        self.N_SEC = rospy.get_param("~N_SEC")
        self.FS = rospy.get_param("~FS")
        self.LOG = rospy.get_param("~WAV_FILE")
        self.STEP_SIZE_1 = rospy.get_param("~STEP_SIZE_1")
        self.STEP_SIZE_2 = rospy.get_param("~STEP_SIZE_2")
        self.MAX_DIST_TRAVEL = rospy.get_param("~MAX_DIST_TRAVEL")
        self.ANGLE_WIDTH = 15. #10#This is the minimum angular margin to each side when finding the largest step size in a given direction
        self.STEP_MARGIN = 0.35 #0.25#When finding the largest step size in a given direction, this number is subtracted to ensure that the robot does not hit anything
        self.DO_PLOT = rospy.get_param("~DO_PLOT")
        self.MAX_IDX = np.floor(M_SPACE/C * self.FS)
        self.Node_name = rospy.get_name()
        file_date = time.strftime("%y_%m_%d_%H_%M_%S")
        self.FILE_PATH = os.path.join(self.LOG,'IMPROVED3_'+file_date)
        os.makedirs(self.FILE_PATH)
        shutil.copyfile(__file__,os.path.join(self.FILE_PATH,os.path.basename(__file__)))
        self.T0 = rospy.get_time()
        rospy.set_param('/T0',self.T0)
        self.RESCALE_POWER = False


        ################## seting initial position
        rospy.loginfo('Setting up initial position and move_base')
        self.ROBOT_LOG = os.path.join(self.FILE_PATH,'Robot')
        os.makedirs(self.ROBOT_LOG)
        self.Robot_Action = RobotMove('/Robot2/mobile_base/commands/velocity','/Robot2/odom',os.path.join(self.ROBOT_LOG,'log_robot.txt'),os.path.join(self.ROBOT_LOG,'log_robot_nav.txt'))
        rospy.loginfo('Done')


        rospy.loginfo('Setting up laser scanner')
        self.LASER_LOG = os.path.join(self.FILE_PATH,'Laser')
        os.makedirs(self.LASER_LOG)
        self.Cost_Map = CostMap('/Robot2/scan_filtered',1.5,os.path.join(self.LASER_LOG,'log_laser.txt'))
        rospy.loginfo('Done')     

        # ################## TTS Audio
        rospy.loginfo('Setting up TTS audio')
        self.TTS = Text2Speech('/Robot2/say_text')
        rospy.loginfo('Done')

        ################## Microphone array Audio
        rospy.loginfo('Setting up recording') 
        self.AUDIO_LOG = os.path.join(self.FILE_PATH,'Audio')
        os.makedirs(self.AUDIO_LOG)       
        self.MicArray = AudioRecording(self.DEV_BACK,self.DEV_FRONT,self.AUDIO_LOG)
        rospy.loginfo('Done')

        rospy.loginfo('Setting up sensor objects')
        self.Front_Mic = AudioProcessing(os.path.join(self.AUDIO_LOG,'log_front.txt'),True) #Last argument determines direction. Used for mapping SSL to angle
        self.Back_Mic = AudioProcessing(os.path.join(self.AUDIO_LOG,'log_back.txt'),False) #Last argument determines direction. Used for mapping SSL to angle
        ##################################

        
        rospy.loginfo('Initialization successful. Node: '+self.Node_name)
        rospy.sleep(6.)
        rospy.on_shutdown(self.shutdown)
        self.process()



    def process(self):
        # Main process for handling everything.
        global ENERGY_INTERVAL
        count = 0
        while not rospy.is_shutdown(): #While loop running forever
            rospy.loginfo('AUDIO Iteration: '+str(count))
            #rospy.loginfo('Calling')
            self.TTS.say_text('Where are you')
            #rospy.loginfo('Done')
            # Record audio
            #rospy.loginfo('Recording ')
            X_back,X_front = self.MicArray.record_audio(self.N_SEC,count)
            #rospy.loginfo('Done ')


            #rospy.loginfo('Calling')
            self.TTS.say_text('I heard you.')
            #rospy.loginfo('Done')

            # Compute features
            temp_feat_back = self.Back_Mic.compute_features(X_back)
            temp_feat_front = self.Front_Mic.compute_features(X_front)

            rospy.loginfo('DOA-BACK: '+str(temp_feat_back[2]))

            norm_features_front = np.zeros((temp_feat_front.shape))
            norm_features_front[2] = combine_doa(temp_feat_front[2],temp_feat_back[2])
            norm_features_front[0:2] = normalize_features(temp_feat_front[0:2])
            fused_feature_front = np.mean(norm_features_front[0:2]) #Fuse

            rospy.loginfo('Normalized Features: ')
            rospy.loginfo('Energy: '+str(norm_features_front[0])+' - DRR: '+str(norm_features_front[1])+' - DOA: '+str(norm_features_front[2])+' - Fused: '+str(fused_feature_front))


            norm_features = norm_features_front
            fused_feature = fused_feature_front
            if np.abs(norm_features[2]) > 89.: #Turn around so the valid range of the laser scanner is aligned with the direction of audio, i.e. the laser scanner does not cover back plane
                self.Robot_Action.rotate(norm_features[2])
                # Map audio angle to front and then continue
                norm_features[2] = 0#map_angle2front(norm_features[2])
                   

            if self.RESCALE_POWER and (count == 0):
                rospy.loginfo('Scaling Energy Interval')
                ENERGY_INTERVAL[0] = norm_features[0]


            self.Robot_Action.write_log(norm_features,'FRONT')
            self.Robot_Action.print_log()


            if np.min(norm_features[0:2]) > FEATURE_THRESHOLD:
                rospy.loginfo('Very close to the source. Moving in that direction and terminating.')
                if self.DO_PLOT:
                        self.Cost_Map.plot_scan()
                self.Robot_Action.make_move1(norm_features[2],0)
                step_size = np.max([self.Cost_Map.get_range(0,self.ANGLE_WIDTH)-self.STEP_MARGIN,0])#

                while step_size > .6: #Move until obstacle is reached
                    self.Robot_Action.make_move1(0,np.max([step_size-.4,0]))
                    step_size = np.max([self.Cost_Map.get_range(0,self.ANGLE_WIDTH)-self.STEP_MARGIN,0])

                self.Robot_Action.make_move1(0,np.max([step_size-.4,0]))

                self.Robot_Action.write_log(np.zeros((3,)),'Finish')
                self.goal_reached()


            else:
                dist_travel = 0 #Accumulate driving distance
                CONFUSED_FLAG = False #Used to indicate if a point of confusion has been reached
                sub_count = 0 #Counter used for navigation step between hearing audio
                prev_gap_size = 0 #Holds the gap size of previous gap. Only used when a single gap is detected
                while (dist_travel < self.MAX_DIST_TRAVEL) and (not CONFUSED_FLAG):
                    rospy.loginfo('NAV Iteration: '+str(sub_count+1)+' - Distance traveled: '+str(dist_travel))
                    
                    # Get gaps, center and size from range measurements
                    gap_start_angle,gap_size,gap_center = self.Cost_Map.get_vg()#Will return 0,0,0 if no gaps are detected
                    
                    if gap_size.all() == 0:
                        N_gaps = 0
                    else:
                        N_gaps = gap_start_angle.shape[0] #len(gap_start_angle)
                    if self.DO_PLOT:
                        self.Cost_Map.plot_scan()
                    
                    self.Robot_Action.write_log_nav(np.array([sub_count,N_gaps,dist_travel]))
                    # We must dinstinguish between the first iteration and the following, because the first one
                    # always comes after having called out for speech, which was caused by reaching a point of confusion, so
                    # the robot must first escape from this/be clear of this point
                    if sub_count == 0: #To escape the point of confusion
                        #rospy.loginfo('First Iter')
                        if N_gaps == 0:
                            if self.Cost_Map.get_range(norm_features[2],self.ANGLE_WIDTH) < 0.4: #Reflection
                                rospy.loginfo('Reflection detected')
                                #step_size = np.min([0.3,self.Cost_Map.get_range(0,self.ANGLE_WIDTH)])
                                self.Robot_Action.make_move1(0,-0.2)
                                self.Robot_Action.make_move1(-180.,0)
                            else:
                                step_size = np.max([np.min([self.Cost_Map.get_range(norm_features[2],self.ANGLE_WIDTH)-self.STEP_MARGIN,self.STEP_SIZE_1]),0])
                                self.Robot_Action.make_move1(norm_features[2],step_size)
                                dist_travel += step_size
                        elif N_gaps == 1:                           
                            if self.Cost_Map.get_range(norm_features[2],self.ANGLE_WIDTH) < 0.4: #Reflection
                                step_size = np.max([np.min([self.Cost_Map.get_range(gap_center[0]*180/np.pi,self.ANGLE_WIDTH)-self.STEP_MARGIN,self.STEP_SIZE_1]),0])
                                temp_angle = gap_center[0]*180/np.pi
                            else:
                                step_size = np.max([np.min([self.Cost_Map.get_range(norm_features[2],self.ANGLE_WIDTH)-self.STEP_MARGIN,self.STEP_SIZE_1]),0])
                                temp_angle = norm_features[2]
                            dist_travel += step_size #self.STEP_SIZE
                            self.Robot_Action.make_move1(temp_angle,step_size)#self.STEP_SIZE)
                        else:
                            #Map doa to gap and go there
                            temp_angle = gap_center[map_angle2gap(norm_features[2],gap_start_angle*180/np.pi,gap_size*180/np.pi)]*180/np.pi
                            step_size = np.max([np.min([self.Cost_Map.get_range(temp_angle,self.ANGLE_WIDTH)-self.STEP_MARGIN,self.STEP_SIZE_1]),0])#np.max([self.Cost_Map.get_range(temp_angle,self.ANGLE_WIDTH)-self.STEP_MARGIN,0])
                            self.Robot_Action.make_move1(temp_angle,step_size)
                            dist_travel += step_size
                    else:
                        #rospy.loginfo('Later Iter')
                        if N_gaps == 0:
                            CONFUSED_FLAG = True
                        elif N_gaps == 1:
                            if sub_count > 1: #This construction is to ensure that there has been two iterations with 1 gap, otherwise it is not possible to compare
                                rospy.loginfo('Gap change: '+str(np.abs(gap_size-prev_gap_size)/prev_gap_size))
                                if np.abs(gap_size-prev_gap_size)/prev_gap_size > GAP_SIZE_THRES: #If the gap size changes Type 2
                                    CONFUSED_FLAG = True
                                    prev_gap_size = gap_size
                                    dist_travel += self.STEP_SIZE_2
                                    #self.Robot_Action.make_move(gap_center*180/np.pi,self.STEP_SIZE_2)
                                    self.Robot_Action.make_move1(gap_center*180/np.pi,self.STEP_SIZE_2)
                                else:
                                    prev_gap_size = gap_size
                                    dist_travel += self.STEP_SIZE_2
                                    #self.Robot_Action.make_move(gap_center*180/np.pi,self.STEP_SIZE_2)
                                    self.Robot_Action.make_move1(gap_center*180/np.pi,self.STEP_SIZE_2)
                            else:
                                prev_gap_size = gap_size
                                dist_travel +=self.STEP_SIZE_2
                                #self.Robot_Action.make_move(gap_center*180/np.pi,self.STEP_SIZE_2)
                                self.Robot_Action.make_move1(gap_center*180/np.pi,self.STEP_SIZE_2)
                        else: #Type 1
                            CONFUSED_FLAG = True

                    sub_count += 1


            #rospy.sleep(1.)
            count += 1

    def goal_reached(self):
        self.TTS.say_text('I believe I have found you. I will terminate.')
        while True and (not rospy.is_shutdown()):
            rospy.sleep(1)
            rospy.loginfo('Sleeping') 

    def shutdown(self):
        rospy.loginfo('Saving map data')
        rospy.sleep(3.)
        newest_file = max(glob.iglob('/home/sociobot/catkin_ws/src/misc/map_data/*.bag'), key=os.path.getctime)
        dst_file = os.path.join(self.FILE_PATH,os.path.basename(newest_file))
        dst_file1 = os.path.join(self.FILE_PATH,'bag_data.bag')
        rospy.loginfo(newest_file)
        rospy.loginfo(dst_file)
        shutil.copyfile(newest_file,dst_file)
        shutil.copyfile(dst_file,dst_file1) #We also want a copy with a generic name. Ideally should use syḿlink

class Text2Speech():
    def __init__(self,service):
        self.service_topic = service
        rospy.loginfo('Waiting for audio service')
        rospy.wait_for_service(self.service_topic)
        rospy.loginfo('Done')
        self.tts_service = rospy.ServiceProxy(self.service_topic, SayText)

    def say_text(self,say_string):
        self.tts_service(say_string)


class AudioRecording:
    def __init__(self,device_back,device_front,data_path):
        rospy.loginfo('Waiting for audio service')
        rospy.wait_for_service('/Robot2/get_audio_segment_duo')
        rospy.loginfo('Done')
        self.device_back = device_back
        self.device_front = device_front
        self.data_path = data_path
        self.get_audio_segment_duo = rospy.ServiceProxy('/Robot2/get_audio_segment_duo', GetAudioSegmentDuo)

    def record_audio(self,N_sec,count): #Creates a unix command string and then uses a service to execute on the robot. Very dirty, but works
        file_name1 = os.path.join(self.data_path,'audio_back_'+str(count)+'.wav')
        file_name2 = os.path.join(self.data_path,'audio_front_'+str(count)+'.wav')
        command1 = 'arecord -D '+self.device_back+' -f cd -r 16000 -c 4 -d '+str(N_sec)+' '+file_name1
        command2 = 'arecord -D '+self.device_front+' -f cd -r 16000 -c 4 -d '+str(N_sec)+' '+file_name2
        #rospy.loginfo('Recording Audio')
        resp1 = self.get_audio_segment_duo(command1,file_name1,count,self.data_path,command2,file_name2,count,self.data_path)
        X1 = np.vstack((resp1.m1_ch1,resp1.m1_ch2,resp1.m1_ch3,resp1.m1_ch4)).T
        X2 = np.vstack((resp1.m2_ch1,resp1.m2_ch2,resp1.m2_ch3,resp1.m2_ch4)).T
        #return X/(2.**15)
        return X1,X2

class AudioProcessing:
    def __init__(self,log_file,front):
        self.t0 = rospy.get_param('/T0')
        self.log_file = log_file
        self.front = front
        self.N_channels = 4
        self.win_size = 512
        self.Fs = 16000.
        self.max_idx = np.floor(M_SPACE/C * self.Fs)
        self.Fid = open(self.log_file,'a')
        self.legend = ['Time','Energy','DRR','SSL']
        self.N_features = len(self.legend)-1
        self.Fid.write(" ".join(self.legend))
        self.Fid.write("\n")
        self.Fid.close()
        self.FLAG = 0

    def compute_features(self,X):
        temp_feat_ar = np.zeros((self.N_features,))

        temp_feat_ar[0] = self.compute_energy(X)
        temp_feat_ar[1] = self.compute_drr(X)
        if self.front:
            temp_doa = self.compute_doa(X)
            temp_feat_ar[2] = np.arcsin(temp_doa /float(self.max_idx))
            temp_feat_ar[2] = temp_feat_ar[2]*180./np.pi
        else:
            temp_feat_ar[2] = self.compute_doa_plane(X)

        if self.FLAG == 0:
            self.feat_ar = temp_feat_ar
            self.FLAG = 1
        else:
            self.feat_ar = np.vstack((self.feat_ar,temp_feat_ar))

        self.Fid = open(self.log_file,'a')
        t_sec = rospy.get_time()-self.t0
        self.Fid.write(str(t_sec)+' ')
        np.savetxt(self.Fid,temp_feat_ar.T,fmt='%10.7f',newline=' ')
        self.Fid.write("\n")
        self.Fid.close()
        
        return temp_feat_ar

    def time_align_signals(self,X,sample_delay):
        if sample_delay == 0:
            return X 
        elif sample_delay < 0: #Tested with <
            X_align = np.zeros(X.shape)
            X_align[:,0:3] = X[:,0:3]
            X_align[0:-np.abs(sample_delay),3] = X[np.abs(sample_delay):,3]
            return X_align
        else:
            X_align = np.zeros(X.shape)
            X_align[:,0:3] = X[:,0:3]
            X_align[np.abs(sample_delay):,3] = X[0:-np.abs(sample_delay),3]
            return X_align


    def compute_energy(self,X):
        N_FRAME = self.win_size
        N = X.shape[0]
        N_ADV = round(N_FRAME*0.5) #Number of samples to advance
        N_WIN = np.floor((N-N_FRAME)/(N_ADV)) #Number of frames
        E_AR = np.zeros((N_WIN,))

        for kk in range(0,int(N_WIN)):
            if kk==0:
                idx_start = 0
                idx_stop = N_FRAME
            else:
                idx_start = idx_start+N_ADV
                idx_stop = idx_start+N_FRAME

            temp_energy = self.avg_energy(X[idx_start:idx_stop,:])
            #rospy.loginfo('Energy: '+str(temp_energy))
            if temp_energy > E_THRES1:
                E_AR[kk] = temp_energy
            else:
                E_AR[kk] = np.inf

        E_AR = E_AR[~np.isinf(E_AR)]
        
        return 10.*np.log10(np.mean(E_AR)/E_THRES)

    def avg_energy(self,X):
        return np.mean(np.var(X,axis=0)) 

    def compute_drr(self,X):
        # This implementation of computing DRR is based on the MATLAB implementation from
        # M. Jeub et. al. from the paper "Blind Estimation of the Coherent-to-Diffuse Energy Ratio From Noisy Speech Signals" 
        m_dist = M_SPACE
        temp_doa = self.compute_doa_1(X,[0,3],int(np.floor(m_dist/C * self.Fs)))
        X = self.time_align_signals(X,temp_doa)      
        N_FRAME = self.win_size
        N = X.shape[0]
        win = np.hanning(N_FRAME)
        N_ADV = round(N_FRAME*0.25) #Number of samples to advance
        N_WIN = np.floor((N-N_FRAME)/(N_ADV)) #Number of frames
        DRR_AR = np.zeros((N_WIN,))
        ALPHA = 0.85
        A_COH = 0.5
        A_DRR = 0.99
        freq_vec = np.linspace(1,8000,N_FRAME/2.+1)
        coh_model = np.sinc(2*freq_vec*m_dist/C)
        DRR_LIM =(0.01,100)
        SPEECH_COUNT = 0
        for kk in range(0,int(N_WIN)):
            if kk==0:
                idx_start = 0
                idx_stop = N_FRAME
                S1_temp = np.abs(np.fft.rfft(X[idx_start:idx_stop,0]*win))**2
                S2_temp = np.abs(np.fft.rfft(X[idx_start:idx_stop,3]*win))**2
                S12_temp = np.fft.rfft(X[idx_start:idx_stop,0]*win)*np.fft.rfft(X[idx_start:idx_stop,3]*win).conj()
            else:
                idx_start = idx_start+N_ADV
                idx_stop = idx_start+N_FRAME
                S1_temp = ALPHA*S1_temp + (1-ALPHA)*np.abs(np.fft.rfft(X[idx_start:idx_stop,0]*win))**2
                S2_temp = ALPHA*S2_temp + (1-ALPHA)*np.abs(np.fft.rfft(X[idx_start:idx_stop,3]*win))**2
                S12_temp = ALPHA*S12_temp + (1-ALPHA)*np.fft.rfft(X[idx_start:idx_stop,0]*win)*np.fft.rfft(X[idx_start:idx_stop,3]*win).conj()

            denum = np.sqrt(S1_temp*S2_temp+np.finfo(float).eps)
            numerator = S12_temp+np.finfo(float).eps

            if not(kk==0):
                coh_tmp = numerator / denum
                coh_smooth = A_COH*coh_smooth + (1-A_COH) * coh_tmp
                DRR_tmp = (coh_model - np.real((coh_smooth))) / (np.real((coh_smooth)) - 1)
                DRR_tmp[DRR_tmp<DRR_LIM[0]] = DRR_LIM[0]
                DRR_tmp[DRR_tmp>DRR_LIM[1]] = DRR_LIM[1]
                temp_energy = self.avg_energy(X[idx_start:idx_stop,:])


                DRR_smooth = A_DRR*DRR_smooth + (1-A_DRR)*DRR_tmp
            else:
                coh_smooth = numerator / denum
                DRR_tmp = (coh_model - np.real((coh_smooth))) / (np.real((coh_smooth)) - 1)
                DRR_tmp[DRR_tmp<DRR_LIM[0]] = DRR_LIM[0]
                DRR_tmp[DRR_tmp>DRR_LIM[1]] = DRR_LIM[1]
                DRR_smooth = DRR_tmp
            
            DRR_AR[kk] = 10*np.log10(np.mean(DRR_smooth))

           
        return np.mean(DRR_AR)


    def gccPhat(self,x1,x2,max_idx,N_peaks):
        N = len(x1)
        win = np.hamming(N)
        X1 = np.fft.fft(x1*win)
        X2 = np.fft.fft(x2*win)
        S = X1*X2.conj()
        Sabs = np.abs(S)
        SS = S/Sabs
        Phi = np.real(np.fft.fftshift(np.fft.ifft(SS)))
        Phi_lim = Phi[N/2-max_idx : N/2+max_idx+1]
        idx_delay = np.argsort(Phi_lim)[-N_peaks:] - max_idx
        return idx_delay

    def compute_doa_plane(self,X): #Returns 1 if more pos than negative delays, otherwise -1
        max_idx = self.max_idx
        N = X.shape[0]
        N_FRAME = self.win_size #Frame lenght
        N_ADV = round(N_FRAME*0.25) #Number of samples to advance
        N_WIN = np.floor((N-N_FRAME)/(N_ADV)) #Number of frames
        DOA_AR = np.zeros((N_WIN,))

        for kk in range(0,int(N_WIN)):
            if kk==0:
                idx_start = 0
                idx_stop = N_FRAME
            else:
                idx_start = idx_start+N_ADV
                idx_stop = idx_start+N_FRAME

            X1_temp = X[idx_start:idx_stop,0]
            X2_temp = X[idx_start:idx_stop,3]
            temp_energy = self.avg_energy(X[idx_start:idx_stop,:])

            if temp_energy > E_THRES1*2:
                DOA_AR[kk] = self.gccPhat(X1_temp,X2_temp,max_idx,1) #Order of input to match the angular convention
            else:
                DOA_AR[kk] = np.inf

        DOA_AR = DOA_AR[~np.isinf(SSL_AR)]
        sum_pos = np.sum(DOA_AR>=0)
        sum_neg = np.sum(DOA_AR<0)

        rospy.loginfo('DOA: Pos: '+str(sum_pos)+' - Neg: '+str(sum_neg))
        if sum_pos >= sum_neg:
            return 1
        else:
            return -1


    def compute_doa(self,X):
        max_idx = self.max_idx
        N = X.shape[0]
        N_FRAME = self.win_size #Frame lenght
        N_ADV = round(N_FRAME*0.25) #Number of samples to advance
        N_WIN = np.floor((N-N_FRAME)/(N_ADV)) #Number of frames
        DOA_AR = np.zeros((N_WIN,))

        for kk in range(0,int(N_WIN)):
            if kk==0:
                idx_start = 0
                idx_stop = N_FRAME
            else:
                idx_start = idx_start+N_ADV
                idx_stop = idx_start+N_FRAME

            X1_temp = X[idx_start:idx_stop,0]
            X2_temp = X[idx_start:idx_stop,3]
            temp_energy = self.avg_energy(X[idx_start:idx_stop,:])

            if temp_energy > E_THRES1:
                DOA_AR[kk] = self.gccPhat(X1_temp,X2_temp,max_idx,1) #Order of input to match the angular convention
            else:
                DOA_AR[kk] = np.inf

        DOA_AR = DOA_AR[~np.isinf(DOA_AR)]
        hist,bin_edges = np.histogram(DOA_AR,bins=np.arange(-max_idx-1,max_idx+2))
        #print hist
        return bin_edges[np.argmax(hist)]#np.var(SSL_AR),hist,bin_edges,bin_edges[np.argmax(hist)]

    def compute_doa_1(self,X,mic_idx,max_idx):
        N = X.shape[0]
        N_FRAME = self.win_size #Frame lenght
        N_ADV = round(N_FRAME*0.25) #Number of samples to advance
        N_WIN = np.floor((N-N_FRAME)/(N_ADV)) #Number of frames
        DOA_AR = np.zeros((N_WIN,))

        for kk in range(0,int(N_WIN)):
            if kk==0:
                idx_start = 0
                idx_stop = N_FRAME
            else:
                idx_start = idx_start+N_ADV
                idx_stop = idx_start+N_FRAME

            X1_temp = X[idx_start:idx_stop,mic_idx[0]]
            X2_temp = X[idx_start:idx_stop,mic_idx[1]]
            temp_energy = self.avg_energy(X[idx_start:idx_stop,:])

            if temp_energy > E_THRES1:
                DOA_AR[kk] = self.gccPhat(X1_temp,X2_temp,max_idx,1) #Order of input to match the angular convention
            else:
                DOA_AR[kk] = np.inf

        DOA_AR = DOA_AR[~np.isinf(DOA_AR)]
        hist,bin_edges = np.histogram(DOA_AR,bins=np.arange(-max_idx,max_idx))
        #print hist
        return bin_edges[np.argmax(hist)]#np.var(SSL_AR),hist,bin_edges,bin_edges[np.argmax(hist)]


class RobotMove:
    def __init__(self,move_topic,odom_topic,log_file,log_file_nav):
        self.t0 = rospy.get_param('/T0') #rospy.get_time()
        self.move_topic = move_topic
        self.odom_topic = odom_topic
        self.move_pub = rospy.Publisher(self.move_topic,Twist,queue_size=1)
        self.angle_speed = 20.*np.pi/180. #Rotational speed
        self.linear_speed = 0.2 #0.2 0.15 #Speed when driving straight
        self.log_file = log_file
        self.log_file_nav = log_file_nav
        self.Fid = open(self.log_file,'a')
        self.legend = ['Time','X','Y','Angle','Norm. Features']
        self.Fid.write(" ".join(self.legend))
        self.Fid.write("\n")
        self.Fid.close()

        self.Fid_nav = open(self.log_file_nav,'a')
        self.legend_nav = ['Time','X','Y','Counter','Gaps','Dist. Travel']
        self.Fid_nav.write(" ".join(self.legend_nav))
        self.Fid_nav.write("\n")
        self.Fid_nav.close()
        self.bumper_flag = False
        rospy.Subscriber('/Robot2/mobile_base/events/bumper',BumperEvent,self.cb_bumper)



    def cb_bumper(self,data):
        if data.state == 1:
            self.bumper_flag = True


    def write_log(self,features,front_back):
        angle = self.get_orientation()
        pos = self.get_position()
        t_sec = rospy.get_time()-self.t0
        w_vec = np.array([t_sec, pos[1], pos[0], angle])
        w_vec_full = np.hstack((w_vec,features))
        self.Fid = open(self.log_file,'a')
        np.savetxt(self.Fid,w_vec_full,fmt='%10.2f',newline=' ')
        self.Fid.write(' '+front_back)
        self.Fid.write("\n")
        self.Fid.close()

    def write_log_nav(self,features):
        angle = self.get_orientation()
        pos = self.get_position()
        t_sec = rospy.get_time()-self.t0
        w_vec = np.array([t_sec, pos[1], pos[0], angle])
        w_vec_full = np.hstack((w_vec,features))
        self.Fid_nav = open(self.log_file_nav,'a')
        np.savetxt(self.Fid_nav,w_vec_full,fmt='%10.2f',newline=' ')
        self.Fid_nav.write("\n")
        self.Fid_nav.close()

    def print_log(self):
        angle = self.get_orientation()
        pos = self.get_position()
        t_sec = rospy.get_time()-self.t0
        rospy.loginfo('Robot info: ')
        rospy.loginfo('Time: '+str(t_sec)+' - Pos, x: '+str(pos[1])+' - Pos, y: '+str(pos[0])+' - Heading: '+str(angle))


    def get_orientation(self):
        msg_angle = rospy.wait_for_message(self.odom_topic,Odometry, timeout=10) #Get odometry
        quat = (msg_angle.pose.pose.orientation.x, msg_angle.pose.pose.orientation.y, msg_angle.pose.pose.orientation.z, msg_angle.pose.pose.orientation.w)
        eul = euler_from_quaternion(quat)
        angle_off = eul[2]*180./np.pi
        return angle_off

    def get_position(self):
        msg_angle = rospy.wait_for_message(self.odom_topic,Odometry, timeout=10) #Get odometry
        return -msg_angle.pose.pose.position.y,msg_angle.pose.pose.position.x


    def rotate(self,angle):
        #Simple closed-loop controller for turning angle degrees
        init_angle = self.get_orientation()
        target_angle = self.handle_angle_wrap(init_angle+angle)
        #abs_angle = np.abs(angle)
        #cur_angle = init_angle
        delta_angle = angle
        msg_twist = Twist()
        #rospy.loginfo('Init - delta_angle: '+str(delta_angle))

        while np.abs(delta_angle) > 1.:
            #rospy.loginfo('Rotating - delta_angle: '+str(delta_angle))
            msg_twist.angular.z = np.sign(angle)*self.angle_speed
            self.move_pub.publish(msg_twist)
            delta_angle = self.handle_angle_wrap(target_angle-self.get_orientation())

        #To stop it again
        for ii in range(5):
            msg_twist.angular.z = 0
            self.move_pub.publish(msg_twist)       

        return delta_angle


    def drive(self,dist):
        # Closed-loop controller for driving the robot straight. 
        init_pos = self.get_position()
        delta_dist = 0
        msg_twist = Twist()
        while delta_dist < dist and (not self.bumper_flag):
            #rospy.loginfo('Driving - delta_dist: '+str(delta_dist))
            #rospy.loginfo('Driving - Pos: '+str(self.get_position()))
            #rospy.loginfo('Driving - Orientation: '+str(self.get_orientation()))
            msg_twist.linear.x = np.sign(dist)*self.linear_speed
            self.move_pub.publish(msg_twist)
            cur_pos = self.get_position()
            delta_dist = np.sqrt((cur_pos[0]-init_pos[0])**2 + (cur_pos[1]-init_pos[1])**2)

        #To stop it again
        for ii in range(1):
            msg_twist.linear.x = 0
            self.move_pub.publish(msg_twist) 


        if self.bumper_flag: # If bumper was activated during driving, then the robot must back up
            init_pos1 = self.get_position()
            target_dist = 0.6
            delta_dist1 = 0
            prev_pos = self.get_position()[1]
            while delta_dist1 < target_dist:
                #rospy.loginfo('Backing - delta_dist: '+str(delta_dist1))
                msg_twist.linear.x = -self.linear_speed*0.5
                self.move_pub.publish(msg_twist)
                cur_pos1 = self.get_position()
                delta_dist1 = np.sqrt((cur_pos1[0]-init_pos1[0])**2 + (cur_pos1[1]-init_pos1[1])**2)


        self.bumper_flag = False
        return np.abs(dist-delta_dist)



    def make_move1(self,angle,distance):
        rospy.loginfo('Robot-Move 1: Angle: '+str(angle)+' - Distance: '+str(distance))
        self.rotate(angle)
        self.drive(distance)



    def handle_angle_wrap(self,angle):
        # Function for wrapping angles to (-180;180]
        angle = angle % 360.
        angle = (angle+360.) % 360.
        if angle > 180.:
            angle -= 360.

        return angle


class CostMap:
    def __init__(self,laser_topic,max_range,log_file):
        self.t0 = rospy.get_param('/T0')
        self.log_file = log_file
        self.laser_topic = laser_topic
        self.max_range = max_range
        self.DO_LOG = True
        self.DO_FILTER = True
        self.Filt_order = 16
        self.min_angle = -np.pi/2
        self.max_angle = np.pi/2
        self.N_scans = 1
        self.MIN_GAP_SIZE = 15.*np.pi/180. 


    def get_scan_raw(self):
        #rospy.loginfo('Scanning....')
        for ii in range(self.N_scans): #Possibly smooth over many consecutive scans
            cur_scan = rospy.wait_for_message(self.laser_topic,LaserScan,timeout=5.)
            cur_scan.ranges = np.asarray(cur_scan.ranges)
            if ii == 0:
                scan_mat = np.zeros((self.N_scans,len(cur_scan.ranges)))

            if np.sum(np.isnan(cur_scan.ranges))>0: #Interpolates all the nan values
                #rospy.loginfo('CostMap: Interpolating NaN values')
                nans, x = nan_helper(cur_scan.ranges)
                cur_scan.ranges[nans]= np.interp(x(nans), x(~nans), cur_scan.ranges[~nans])
            cur_scan.ranges[cur_scan.ranges>self.max_range] = self.max_range
            cur_scan.ranges[0] = 0
            cur_scan.ranges[-1] = 0

            scan_mat[ii,:] = cur_scan.ranges
            if self.N_scans > 1:
                rospy.sleep(0.02)
        
        cur_scan.ranges = np.mean(scan_mat,axis=0)
        return cur_scan

    def get_scan(self):
        cur_scan = self.get_scan_raw()
        
        self.min_angle = cur_scan.angle_min
        self.max_angle = cur_scan.angle_max

        if self.DO_FILTER: #Apply moving average
            w = np.ones(self.Filt_order,'d')
            cur_scan.ranges=np.convolve(w/w.sum(),cur_scan.ranges,mode='same')

        if self.DO_LOG:
            t_sec = rospy.get_time()-self.t0
            self.Fid = open(self.log_file,'a')
            self.Fid.write(str(round(t_sec,2))+' ')
            np.savetxt(self.Fid,cur_scan.ranges,fmt='%10.2f',newline=' ')            
            self.Fid.write("\n")            
            self.Fid.close()

        return cur_scan

    def get_zero_runs(self,a):
        # Create an array that is 1 where a is 0, and pad each end with an extra 0.
        iszero = np.concatenate(([0], np.equal(a, 0).view(np.int8), [0]))
        absdiff = np.abs(np.diff(iszero))
        # Runs start and end where absdiff is 1.
        ranges = np.where(absdiff == 1)[0].reshape(-1, 2)
        return ranges


    def get_vg(self):
        cur_scan = self.get_scan()
        ranges = cur_scan.ranges
        N_ranges = len(ranges) 
        angle_ar = 180./np.pi * np.linspace(-np.pi/2.,np.pi/2.,num=len(ranges))   

        win_size = 25
        max_ar = np.ones(N_ranges,)
        ranges_zp = np.hstack((np.zeros(win_size,),ranges,np.zeros(win_size,)))

        #Find local maxima
        for ii in range(win_size,N_ranges+win_size):
            jj_idx = np.arange(ii-win_size,ii+win_size+1)
            if np.sum(ranges[ii-win_size]<ranges_zp[jj_idx])>0:
                max_ar[ii-win_size] = 0


        max_ar_inv = np.equal(max_ar,0)
        VG = angle_ar[self.get_zero_runs(max_ar_inv)]*np.pi/180.



        N_VG = VG.shape[0]
        if N_VG == 0:
            return np.array([0]),np.array([0]),np.array([0])
        else:
            VG_size = np.zeros(N_VG,)
            VG_center = np.zeros(N_VG,)
            VG_start = VG[:,0]
            for ii in range(N_VG):
                VG_size[ii] = VG[ii,1]-VG[ii,0]
                VG_center[ii] = VG_start[ii] + VG_size[ii]/2.

        idx = np.where(VG_size<self.MIN_GAP_SIZE) #Indices of small gaps
        if idx:
            VG_start = np.delete(VG_start,idx)
            VG_center = np.delete(VG_center,idx)
            VG_size = np.delete(VG_size,idx)
            if not VG_start.all(): #We must check if all gaps were removed
                return np.array([0]),np.array([0]),np.array([0])


        return VG_start,VG_size,VG_center



    def get_range(self,angle,width): #Returns the range in direction of angle, while stile maintaining width
        cur_scan = self.get_scan_raw()
        cur_scan.ranges[cur_scan.ranges>self.max_range] = self.max_range
        N_points = cur_scan.ranges.shape[0] 
        angle_ar = 180./np.pi * np.linspace(cur_scan.angle_min,cur_scan.angle_max,num=N_points)
        return np.min(cur_scan.ranges[(angle_ar<(angle+width))*(angle_ar>(angle-width))])


    def plot_scan(self):
        scan_msg = self.get_scan()
        raw_scan_msg = self.get_scan_raw()
        derivative_ranges = self.get_scan_derivative()
        derivative_ranges_filt = derivative_ranges.copy()
        derivative_ranges_filt[np.abs(derivative_ranges_filt)<0.001] = 0
        angle_ar = 180./np.pi * np.linspace(scan_msg.angle_min,scan_msg.angle_max,num=len(scan_msg.ranges))

        plt.figure()
        plt.plot(angle_ar,scan_msg.ranges)
        plt.title('Filtered')
        plt.show()



def main(args):
    rospy.init_node('find_sound_source')
    #print prev_Phi.shape
    ic = findSoundSource() 
    try:
        rospy.spin()
    except rospy.ROSInterruptException:
        pass

if __name__ == '__main__':
    main(sys.argv)
