#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, time
import os
import glob
import subprocess
import shutil
from subprocess import call
from subprocess import check_output
import roslib; 
import rospy
import pickle
import numpy as np
import matplotlib.pyplot as plt
from scipy import signal
import scipy.io.wavfile as sc_wav
import scipy.stats as sc_stat
from std_msgs.msg import Float32
from navigation.msg import RelativeMove
import numpy.linalg as LA
from move_base_msgs.msg import *
from geometry_msgs.msg import *
from actionlib_msgs.msg import *
from actionlib.msg import *
from std_msgs.msg import Float32
import tf
from sound_play.libsoundplay import SoundClient
import wave
import contextlib
from audio_record.srv import *
from tf.transformations import euler_from_quaternion
from nav_msgs.msg import Odometry
from nav_msgs.msg import OccupancyGrid
from sensor_msgs.msg import LaserScan
from kobuki_msgs.msg import BumperEvent
from scipy.stats import multivariate_normal
from mpl_toolkits.axes_grid1 import make_axes_locatable
import scipy
import h5py
from leg_detection.msg import *
from scipy.optimize import minimize
import scipy.spatial.distance as s_dist
from itertools import product


M_SPACE = 0.226 #Spacing between most outer microphones
C = 342 #Speed of sound
RES = 0.05

FONTSIZE = 6
FIG_SIZE = (9./2.54,9./2.54)
DO_SAVE = False
DPI = 500
MS = 2
ARROW_LEN = 0.15


#to see audio devices go to /dev/snd/by-id/

# Various Helping Function:
def rotate_point(x,phi):
    #Function for rotating point. The sign is introduced to match the orientation
    y = np.copy(x)
    y[0] = x[0]*np.cos(-phi) - x[1]*np.sin(-phi)
    y[1] = x[0]*np.sin(-phi) + x[1]*np.cos(-phi)

    return y

def rolling_window(a, window): #Credit to http://stackoverflow.com/questions/27852343/split-python-sequence-time-series-array-into-subsequences-with-overlap
    shape = a.shape[:-1] + (a.shape[-1] - window + 1, window)
    strides = a.strides + (a.strides[-1],)
    return np.lib.stride_tricks.as_strided(a, shape=shape, strides=strides)

def calc_orientation(S,X):
    M,N = X.shape
    angle_ar = np.zeros((M,))

    for mm in range(M):
        angle_ar[mm] = np.arctan2(X[mm,0]-S[0],X[mm,1]-S[1])

    #print angle_ar
    return (np.max(angle_ar)-np.min(angle_ar))/2. + np.min(angle_ar)

def print_pos_est_direction(S,X,S_pose):
    M,N = X.shape
    angle_ar = np.zeros((M,))
    for mm in range(M):
        tmp_angle = np.arctan2(X[mm,0]-S[0],X[mm,1]-S[1])*180/np.pi - S_pose
        rospy.loginfo('Source '+str(mm)+': '+str(tmp_angle))
        angle_ar[mm] = tmp_angle

    return (np.max(angle_ar)-np.min(angle_ar))/2. + np.min(angle_ar)

def fn(x, A, b):
    return np.linalg.norm(A.dot(x) - b)


def validate_positions(Pos_mat,Source_mat):
    Room_size = np.array([4,6,2.5]) #np.array([7,7])
    Source_dist = 0.5

    N_pos = Pos_mat.shape[0]
    N_source = Source_mat.shape[0]

    for dd in range(N_pos):
        tmp_pos = Pos_mat[dd,:]
        if tmp_pos[0] <= 0 or tmp_pos[0] >= Room_size[0] or tmp_pos[1] <= 0 or tmp_pos[1] >= Room_size[1]:
            return False 
        for kk in range(N_source):
            tmp_dist = np.linalg.norm(tmp_pos-Source_mat[kk,:])
            if tmp_dist < Source_dist:
                return False

    return True


def comp_cond_number(Source_pos,listening_pos):
    N_source = Source_pos.shape[0]
    N_pos = listening_pos.shape[0]

    dist_mat = np.zeros((N_pos,N_source))

    for ii in range(N_pos):
        for kk in range(N_source):
            dist_mat[ii,kk] = np.linalg.norm(listening_pos[ii,:]-Source_pos[kk,:])

    A = 1 / (4*np.pi*dist_mat)**2

    return np.linalg.cond(A)



def comp_best_pos(Source_pos,dist,N_pos):
    N_source = Source_pos.shape[0]

    angle_ar = np.linspace(0,2*np.pi,N_pos,endpoint=False)
    x_coor = dist*np.cos(angle_ar)
    y_coor = dist*np.sin(angle_ar)


    if N_source == 2:
        comb_array = list(product(range(N_pos),range(N_pos)))
    if N_source == 3:
        comb_array = list(product(range(N_pos),range(N_pos),range(N_pos)))
    if N_source == 4:
        comb_array = list(product(range(N_pos),range(N_pos),range(N_pos),range(N_pos)))
    if N_source == 5:
        comb_array = list(product(range(N_pos),range(N_pos),range(N_pos),range(N_pos),range(N_pos)))

    N_comb = len(comb_array)
    cond_ar = np.zeros(N_comb,)

    for ii in range(N_comb):
        comb_tmp = comb_array[ii]
        pos_mat = np.zeros((N_source,2))
        for dd in range(N_source):
            pos_mat[dd,0] = x_coor[comb_tmp[dd]]+Source_pos[dd,0]
            pos_mat[dd,1] = y_coor[comb_tmp[dd]]+Source_pos[dd,1]

        cond_ar[ii] = comp_cond_number(Source_pos,pos_mat)

    fig1 = plt.figure(figsize=FIG_SIZE,tight_layout=True)
    ax1_1 = fig1.add_subplot(111)
    ax1_1.set_ylim(1,3)
    ax1_1.plot(cond_ar)

    min_idx = np.argmin(cond_ar)
    idx_sort = np.argsort(cond_ar) 

    for kk in range(N_comb):
        best_comb = comb_array[idx_sort[kk]]
        best_pos_mat = np.zeros((N_source,2))

        for dd in range(N_source):
            best_pos_mat[dd,0] = x_coor[best_comb[dd]]+Source_pos[dd,0]
            best_pos_mat[dd,1] = y_coor[best_comb[dd]]+Source_pos[dd,1]


        ret_val = validate_positions(best_pos_mat,Source_pos)
        if ret_val:
            return best_pos_mat,cond_ar[idx_sort[kk]]

    return False,False

class SirOptNav:
    def __init__(self):
        self.Node_name = rospy.get_name()
        self.METHOD = rospy.get_param("~METHOD")
        self.AUDIO_DEV = rospy.get_param("~AUDIO_DEV")
        self.AUDIO_DEV_SINGLE = rospy.get_param("~AUDIO_DEV_SINGLE")
        self.N_SEC = rospy.get_param("~N_SEC")
        self.FS = rospy.get_param("~FS")
        self.LOG = rospy.get_param("~WAV_FILE")
        self.STEP_SIZE = rospy.get_param("~STEP_SIZE")
        self.DO_PLOT = rospy.get_param("~DO_PLOT")
        self.MIN_ITER = rospy.get_param("~POS_EST_MIN_ITER")
        self.POS_CONV_THRES = rospy.get_param("~POS_CONV_THRES")
        if self.POS_CONV_THRES < 0:
            self.POS_CONV_THRES = 0
        self.N_sources = rospy.get_param("~N_SOURCES")
        self.FD_node = rospy.get_param("~FaceDetectionNode")
        self.LD_node = rospy.get_param("~LegDetectionNode")
        self.RES = RES
        self.Room_dim = (4,6)
        self.N_points = [int((self.Room_dim[0]-self.RES)/self.RES),int((self.Room_dim[1]-self.RES)/self.RES)]

        self.MAX_IDX = np.floor(M_SPACE/C * self.FS)
        file_date = time.strftime("%y_%m_%d_%H_%M_%S")
        self.FILE_PATH = os.path.join(self.LOG,self.METHOD+'_'+file_date)
        os.makedirs(self.FILE_PATH)
        shutil.copyfile(__file__,os.path.join(self.FILE_PATH,os.path.basename(__file__)))
        self.T0 = rospy.get_time()
        rospy.set_param('/T0',self.T0)


        self.MIC_ARRAY = np.array([[-0.113, 0, 0],[0.113, 0, 0]])


        ################## seting initial position
        rospy.loginfo('Setting up initial position and move_base')
        self.ROBOT_LOG = os.path.join(self.FILE_PATH,'Robot')
        os.makedirs(self.ROBOT_LOG)
        self.Robot_Action = RobotMove([2.,1.],self.STEP_SIZE,self.MIC_ARRAY,self.Room_dim,'/mobile_base/commands/velocity','/odom','/move_base',self.ROBOT_LOG)
        rospy.loginfo('Done')
   
        #################### Cost function
        rospy.loginfo('Setting up Cost function')
        self.CostFunc_LOG = os.path.join(self.FILE_PATH,'CostFunction')
        os.makedirs(self.CostFunc_LOG)
        self.CostFunction = CostFunction(self.Room_dim,self.CostFunc_LOG)

        ################## Microphone array Audio
        rospy.loginfo('Setting up recording') 
        self.AUDIO_LOG = os.path.join(self.FILE_PATH,'Audio')
        os.makedirs(self.AUDIO_LOG)       
        self.MicArray = AudioRecording(self.AUDIO_DEV,self.AUDIO_DEV_SINGLE,self.AUDIO_LOG)
        rospy.loginfo('Done')

        rospy.loginfo('Setting up TDOA')
        self.TDOA_LOG = os.path.join(self.FILE_PATH,'TDOA')
        os.makedirs(self.TDOA_LOG)
        self.Tdoa = TDOA(self.MIC_ARRAY,self.N_sources,self.TDOA_LOG)

        rospy.loginfo('Setting up Energy Estimation')
        self.ENERGY_LOG = os.path.join(self.FILE_PATH,'Energy')
        os.makedirs(self.ENERGY_LOG)
        self.Energy = ENERGY(self.ENERGY_LOG)

        rospy.loginfo('Setting up Position Estimation')
        self.POSITION_LOG = os.path.join(self.FILE_PATH,'Position')
        os.makedirs(self.POSITION_LOG)
        self.X_grid = np.linspace(self.RES,self.Room_dim[0]-self.RES,self.N_points[0])
        self.Y_grid = np.linspace(self.RES,self.Room_dim[1]-self.RES,self.N_points[1])
        self.Pos_Est = PosEstimation(self.X_grid,self.Y_grid,0.01/C,self.POSITION_LOG)

        rospy.loginfo('Setting up Source Classification')
        self.Source_classification_LOG = os.path.join(self.FILE_PATH,'SOURCE_CLASSIFICATION')
        os.makedirs(self.Source_classification_LOG)
        self.SrcClass = SourceClassification(self.FD_node,self.X_grid,self.Y_grid,self.Source_classification_LOG)

        rospy.loginfo('Setting up Source Classification')
        self.Source_classification_laser_LOG = os.path.join(self.FILE_PATH,'SOURCE_CLASSIFICATION_LASER')
        os.makedirs(self.Source_classification_laser_LOG)
        self.SrcClassLaser = SourceClassificationLaser(self.LD_node,self.Room_dim,self.X_grid,self.Y_grid,self.Source_classification_laser_LOG)
        ##################################

        
        rospy.loginfo('Initialization successful. Node: '+self.Node_name)
        rospy.sleep(5.)

        if self.METHOD == 'SOURCE_CLASSIFICATION_TEST':
            self.source_classification_test()
        elif self.METHOD == 'SIR_MEASURE':
            self.measure_SIR_1()
        else:
            self.process()

    def record_test(self):
        for ii in range(4):
            rospy.loginfo('#Iter: '+str(ii+1)+' Recording Audio')
            X,X_close = self.MicArray.record_audio(self.N_SEC)
            X_pow_array = 10*np.log10(np.mean(X**2,axis=0))
            X_pow = 10*np.log10(np.mean(X_close**2))
            rospy.loginfo(str(X_close.shape))
            rospy.loginfo(str(X_pow_array))
            rospy.loginfo(str(X_pow))
            rospy.loginfo(str(np.min(X))+' '+str(np.max(X)))
            rospy.loginfo(str(np.min(X_close))+' '+str(np.max(X_close)))
            plt.plot(X_close)
            plt.show()
            inp = raw_input('Press "q" to quit')
    

    def odom_test(self):
        while True:
            cur_rob_state = self.Robot_Action.get_state()
            rospy.loginfo('Pose: '+str(cur_rob_state[2]))
            inp = raw_input('Press "q" to quit')
            if inp=='q':
                break
            self.Robot_Action.make_move2(20,0,-20)

    def send_goal_test(self):
        raw_input('Press a key to begin!')
        Goal_mat = np.array([[3,1],[3,2],[3,5],[1,1]])
        N_goals = Goal_mat.shape[0]

        for ii in range(N_goals):
            rospy.loginfo('Sending goal: '+str(ii))
            ret_val = self.Robot_Action.move_pos_abs_pos1(Goal_mat[ii,:])
            rospy.loginfo('Success')
            raw_input('Press a key to continue!')

    def move2abs_pos_test(self):
        rospy.loginfo('Absolute Move Test')
        self.Robot_Action.make_move1(np.random.uniform(-15,15,1)[0],np.random.uniform(0.3,2.5,1)[0])
        rospy.sleep(1.)
        self.Robot_Action.move2pos(np.array([3.4,1]),140.)

    def find_start_idx(x,valid_range):
        x_pow = x**2;


    def find_max_seg(self,x,win_len):
        Fs = 16000
        N_samples = x.size
        N_skip = 32
        N_win = int(win_len*Fs)
        N_frames = (N_samples-N_win)/N_skip + 1
        E_ar = np.zeros((N_frames,3))

        for ii in range(N_frames):
            idx1 = ii*N_skip
            idx2 = idx1+N_win+1
            #print idx1,idx2,np.mean(X[idx1:idx2]**2)
            E_ar[ii,0] = np.mean(x[idx1:idx2]**2)
            E_ar[ii,1] = int(idx1)
            E_ar[ii,2] = int(idx2)

        max_idx = np.argmax(E_ar[:,0])
        return E_ar[max_idx,1],E_ar[max_idx,2]

    def measure_SIR(self):
        N_sig = 1 #Number of desired sources
        N_int = 1 #Number of interfering sources
        N_sec = 30 #Recording length when measuring SIR
        N_active = 25 #Seconds of active signal out of N_sec
        Pow_meas_int = np.zeros(N_int,)
        Pow_meas_sig = np.zeros(N_sig,)

        cur_rob_state = self.Robot_Action.get_state()

        # Record Audio
        for ii in range(N_int):
            raw_input('Press a key to record interfering source '+str(ii+1))
            X,X_close = self.MicArray.record_audio(N_sec)
            sc_wav.write(os.path.join(self.FILE_PATH,'Int_'+str(ii+1)+'.wav'),16000,X_close)
             # Energy (Done)
            idx1,idx2 = self.find_max_seg(X_close,N_active)
            Pow_tmp = self.Energy.measure_power1(X_close[idx1:idx2],cur_rob_state)# - self.Energy.measure_power1(X_close[N_active*16000:],cur_rob_state)
            Pow_meas_int[ii] = np.mean(X_close[idx1:idx2]**2)

        for ii in range(N_sig):
            raw_input('Press a key to record desired source '+str(ii+1))
            X,X_close = self.MicArray.record_audio(N_sec)
            sc_wav.write(os.path.join(self.FILE_PATH,'Sig_'+str(ii+1)+'.wav'),16000,X_close)
             # Energy (Done)
            idx1,idx2 = self.find_max_seg(X_close,N_active)
            Pow_tmp = self.Energy.measure_power1(X_close[idx1:idx2],cur_rob_state)# - self.Energy.measure_power1(X_close[N_active*16000:],cur_rob_state)
            Pow_meas_sig[ii] = np.mean(X_close[idx1:idx2]**2)

        SIR_est = Pow_meas_sig[0] / np.sum(Pow_meas_int)
        np.savetxt(os.path.join(self.FILE_PATH,'Pow_sig_meas.txt'),Pow_meas_sig.T,delimiter='')
        np.savetxt(os.path.join(self.FILE_PATH,'Pow_int_meas.txt'),Pow_meas_int.T,delimiter='')
        np.savetxt(os.path.join(self.FILE_PATH,'SIR_meas.txt'),np.array([SIR_est, 10*np.log10(SIR_est)]).T,delimiter='')
        rospy.loginfo('############## DONE ###############')

    def source_classification_test(self):
        rospy.loginfo('Source Classification Test')
        rospy.sleep(3.)
        N_pos = 3
        N_orientation = 3
        N_total = N_pos*N_orientation
        tmp_input = raw_input('Press any key to do human detection')
        N_iter = 3
        for pp in range(N_pos):
            for dd in range(N_orientation):
                self.Source_classification_laser_LOG = os.path.join(self.FILE_PATH,'SOURCE_CLASSIFICATION_LASER_'+str(pp)+'_'+str(dd))
                os.makedirs(self.Source_classification_laser_LOG)
                self.SrcClassLaser = SourceClassificationLaser(self.LD_node,self.Room_dim,self.X_grid,self.Y_grid,self.Source_classification_laser_LOG)   
                tmp_input = raw_input('Go to next pose/pos and press enter to continue')             
                for kk in range(N_iter):
                    cur_rob_state = self.Robot_Action.get_state()
                    rospy.loginfo('Robot Position:'+str(cur_rob_state[0:2])+'  '+str(cur_rob_state[2])) 
                    self.SrcClassLaser.detect_human(cur_rob_state[0:2],cur_rob_state[2])
                    rospy.sleep(0.2)
                    #self.Robot_Action.make_move1(0.,0.2)


                src_pos,pose_vec = self.SrcClassLaser.pos_estimation()
                rospy.loginfo('##########')
                rospy.loginfo(str(src_pos))
                rospy.loginfo(str(pose_vec))

        rospy.loginfo('FINISHED')


    def process(self):
        tmp_input = raw_input('Press any key to start!')

        # Position Estimation
        pos_err = np.ones(self.MIN_ITER,)*100
        pos_err[0] = 100
        pos_count = 0
        CONV_FLAG = False
        while pos_count < self.MIN_ITER and (not CONV_FLAG):
            cur_rob_state = self.Robot_Action.get_state()
            self.Robot_Action.log_state()
            rospy.loginfo('Robot: '+str(cur_rob_state))

            # Face / Body detection
            rospy.loginfo('#Iter: '+str(pos_count+1)+' Leg Pair Detection')
            #tmp_input = raw_input('Press any key! 11111111111')
            self.SrcClassLaser.detect_human(cur_rob_state[0:2],cur_rob_state[2])
            #tmp_input = raw_input('Press any key! 22222222222')

            # Record Audio (Done)
            rospy.loginfo('#Iter: '+str(pos_count+1)+' Recording Audio')
            X,X_close = self.MicArray.record_audio(self.N_SEC)
            #rospy.loginfo(str(X.shape))

            # TDOA + Power (Done)
            rospy.loginfo('#Iter: '+str(pos_count+1)+' TDOA Estimation')
            count,edges,Tau_hat = self.Tdoa.estimate_tdoa(X)

            # Estimate Source Positions (Done)
            rospy.loginfo('#Iter: '+str(pos_count+1)+' POS. Estimation')
            L_mat,Peak_mat,Pos_hat_tmp = self.Pos_Est.gridMlMs(self.Tdoa.Tau_ar,self.Robot_Action.log_state_ar[:,0:2],self.Robot_Action.log_state_ar[:,2]*np.pi/180.,self.MIC_ARRAY)
            # Check convergence of position estimates
            if pos_count > 0:
                pos_est_ar = self.Pos_Est.pos_est_ar
                cur_err = (pos_est_ar[:,:,-1]-pos_est_ar[:,:,-2])**2
                pos_err[pos_count] = np.mean(np.sqrt(np.sum(cur_err,axis=1)))
                rospy.loginfo('Change in position estimate: '+str(pos_err[pos_count]))
                if pos_count > 1:
                    if np.sum(pos_err[pos_count-1:pos_count+1]<self.POS_CONV_THRES):
                        rospy.loginfo('Position Estimation Converged')
                        CONV_FLAG = True

            # Find opt move
            rospy.loginfo('#Iter: '+str(pos_count+1)+' Det. Next Pos.')
            if self.METHOD == 'PROPOSED' or self.METHOD == 'PROPOSED_1':
                opt_move_grid,new_move,new_phi = self.Robot_Action.find_opt_move(self.X_grid,self.Y_grid,Peak_mat,Pos_hat_tmp)
                new_move_r = np.linalg.norm(new_move)
                new_move_phi = np.arctan2(new_move[0],new_move[1])*180./np.pi
                rospy.loginfo('Opt. move: x: '+str(new_move[0])+' - y: '+str(new_move[1]))
                rospy.loginfo('Opt. move: r: '+str(new_move_r)+' - phi: '+str(new_move_phi)+' - new_pose: '+str(new_phi)+' '+str(new_phi*180./np.pi))

            elif self.METHOD == 'BASELINE':
                tmp_rob_state = self.Robot_Action.get_state()
                new_move_phi = print_pos_est_direction(tmp_rob_state[0:2],Pos_hat_tmp,tmp_rob_state[2])
                new_move_r = 0.5
            elif self.METHOD == 'MANUAL':
                tmp_input = raw_input('Move robot and Press any key!')
                new_move_phi = 0
                new_move_r = 0
                
            # If it is the last iteration, then do not move
            if pos_count < self.MIN_ITER-1 and (not CONV_FLAG):# and pos_count > 0:
                # Move to position
                rospy.loginfo('#Iter: '+str(pos_count+1)+' Move to Pos.')
                self.Robot_Action.make_move2(new_move_phi,new_move_r,-new_move_phi)

                # Align heading with the estimate of source positions
                rospy.sleep(0.5)
                tmp_rob_state = self.Robot_Action.get_state()
                new_heading = print_pos_est_direction(tmp_rob_state[0:2],Pos_hat_tmp,tmp_rob_state[2])
                self.Robot_Action.make_move1(new_heading,0.)
                rospy.sleep(0.5)

            
            pos_count += 1

        # Source Classification
        rospy.loginfo('Source Classification')

        NO_HUMAN_FLAG = True
        while NO_HUMAN_FLAG:
            human_idx,human_pose = self.SrcClassLaser.match_source(Pos_hat_tmp)
            if human_idx:
                rospy.loginfo('Human Detection: '+str(human_idx)+' - '+str(Pos_hat_tmp[human_idx,:])+' - '+str(human_pose))
                break
            else:
                rospy.loginfo('No humans detected - moving closer')
                rospy.sleep(0.5)
                tmp_rob_state = self.Robot_Action.get_state()
                new_heading = print_pos_est_direction(tmp_rob_state[0:2],Pos_hat_tmp,tmp_rob_state[2])
                self.Robot_Action.make_move1(new_heading,0.3)
                rospy.sleep(0.5)
                cur_rob_state = self.Robot_Action.get_state()
                self.SrcClassLaser.detect_human(cur_rob_state[0:2],cur_rob_state[2])

        if human_idx:
            rospy.loginfo('Human Detection: '+str(human_idx)+' - '+str(Pos_hat_tmp[human_idx,:])+' - '+str(human_pose))
            interferer_list = range(self.N_sources)
            for hh in range(len(human_idx)):
                interferer_list.remove(human_idx[hh])
        else:
            rospy.loginfo('No humans detected')    


        if self.METHOD == 'PROPOSED':
            # Cost Function Computation
            J_total,opt_pos = self.CostFunction.eval_J_total(Pos_hat_tmp,source_power_hat,human_idx,human_pose)
        elif self.METHOD == 'BASELINE' or self.METHOD == 'PROPOSED_1':
            opt_pos = np.zeros(2,)
            opt_pos[0] = Pos_hat_tmp[human_idx,0] + self.CostFunction.j_soc_dist_param[0]*np.cos(human_pose[human_idx]*np.pi/180.)
            opt_pos[1] = Pos_hat_tmp[human_idx,1] + self.CostFunction.j_soc_dist_param[0]*np.sin(human_pose[human_idx]*np.pi/180.)
            # Move to Optimum Position

        rospy.loginfo('Moving to interaction position')
        new_pos = opt_pos
        rospy.loginfo(str(new_pos))
        self.Robot_Action.move_pos_abs_pos1(new_pos)
        rospy.sleep(0.1)
        cur_rob_state = self.Robot_Action.get_state()
        #self.Robot_Action.log_state()
        rospy.loginfo('Robot: '+str(cur_rob_state))            
        self.Robot_Action.move2pos(new_pos,0.)
        rospy.sleep(0.1)
        cur_rob_state = self.Robot_Action.get_state()
        #self.Robot_Action.log_state()
        rospy.loginfo('Robot: '+str(cur_rob_state))
        new_heading = np.arctan2(Pos_hat_tmp[human_idx,0]-cur_rob_state[0],Pos_hat_tmp[human_idx,1]-cur_rob_state[1])*180./np.pi - cur_rob_state[2]
        #new_heading = print_pos_est_direction(cur_rob_state[0:2],Pos_hat_tmp,cur_rob_state[2])
        rospy.loginfo('New Heading: '+str(new_heading))
        self.Robot_Action.make_move1(new_heading,0.)
        cur_rob_state = self.Robot_Action.get_state()
        self.Robot_Action.log_state()


        rospy.sleep(1.)
        rospy.loginfo('######## DONE MOVING - MEASURE SIR ##########')
        self.measure_SIR()


    def goal_reached(self):
        self.TTS.say_text('I believe I have found you. I will terminate.')
        while True and (not rospy.is_shutdown()):
            rospy.sleep(1)
            rospy.loginfo('Sleeping') 

    def shutdown(self):
        rospy.loginfo('Saving map data')
        rospy.sleep(3.)
        newest_file = max(glob.iglob('/home/sociobot/catkin_ws/src/misc/map_data/*.bag'), key=os.path.getctime)
        dst_file = os.path.join(self.FILE_PATH,os.path.basename(newest_file))
        dst_file1 = os.path.join(self.FILE_PATH,'bag_data.bag')
        rospy.loginfo(newest_file)
        rospy.loginfo(dst_file)
        shutil.copyfile(newest_file,dst_file)
        shutil.copyfile(dst_file,dst_file1) #We also want a copy with a generic name. Ideally should use syḿlink

class CostFunction:
    def __init__(self,Room_dim,LOG_DIR):
        self.t0 = rospy.get_param('/T0')
        self.LOG_DIR = LOG_DIR    
        self.LOG_FILE = os.path.join(self.LOG_DIR,'CostFunction.txt')
        self.DIM = np.asarray(Room_dim)
        self.RES = RES
        self.N_points = ((self.DIM-self.RES)/self.RES).astype(int)
        self.X_grid = np.linspace(self.RES,self.DIM[0]-self.RES,self.N_points[0])
        self.Y_grid = np.linspace(self.RES,self.DIM[1]-self.RES,self.N_points[1])        
        self.XX_grid,self.YY_grid = np.meshgrid(self.X_grid,self.Y_grid)

        #Settings for social costfunction
        #self.j_soc_angle_beta = 8
        #self.j_soc_angle_fov = 120.
        #self.j_soc_dist_beta = 8
        self.j_soc_dist_param = [0.7, 1.3]#[1.2, 1.8] #[0.7, 1.3]
        self.j_soc_angle_param = [-50., 50.] 


    def eval_J_total(self,pos_est,pow_est,source_idxs,pose_est):
        J_SIR = self.eval_J_minSIR(pos_est,pow_est,source_idxs)
        J_social = self.eval_constraint_social(pos_est,source_idxs,pose_est)
        #rospy.loginfo(str(J_social.shape))
        J_total = J_SIR+J_social
        opt_pos = np.zeros(2,)
        pos_idx1 = np.unravel_index(J_total.flatten().argsort()[::-1][0],J_total.shape)
        opt_pos[0] = self.X_grid[pos_idx1[1]]
        opt_pos[1] = self.Y_grid[pos_idx1[0]]
        #rospy.loginfo(str(J_total.shape))
        J_total[J_total<-30] = -30
        J_total[J_total>30] = 30
        if self.LOG_DIR:
            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            surf_tmp = ax1.imshow(np.flipud(J_total),cmap=plt.cm.gray,extent=[self.X_grid[0],self.X_grid[-1],self.Y_grid[0],self.Y_grid[-1]])
            divider = make_axes_locatable(ax1)
            cax = divider.append_axes("right", size="5%", pad=0.05)            
            self.fig.colorbar(surf_tmp,cax=cax)
            ax1.plot(opt_pos[0], opt_pos[1], 'o', ms=MS, color='r')
            #for kk in range(N_interferer):
            #    ax1.plot(pos_est[interferer_idxs[ii],0], pos_est[interferer_idxs[ii],1], 'o', ms=MS, color='r')

            #for kk in range(N_source):
            #    ax1.plot(pos_est[source_idxs[ii],0], pos_est[source_idxs[ii],1], 'o', ms=MS, color='g')
            save_name = os.path.join(self.LOG_DIR,'J_Total.png')
            ax1.tick_params(labelsize=FONTSIZE)
            ax1.set_xlabel('X coordinate',fontsize=FONTSIZE)
            ax1.set_ylabel('Y coordinate',fontsize=FONTSIZE)
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')
            plt.close(self.fig)

            f_name = os.path.join(self.LOG_DIR,'data.h5')
            h5f = h5py.File(f_name, 'w')
            h5f.create_dataset('J_SIR', data=J_SIR)
            h5f.create_dataset('J_social', data=J_social)
            #h5f.create_dataset('J_dist', data=J_dist)
            h5f.create_dataset('pose_estimates', data=pose_est)
            h5f.create_dataset('X_opt', data=opt_pos)
            h5f.close()

        return J_total,opt_pos

    def eval_RSI(self,pos,G0):
        dist = np.sqrt( (self.XX_grid-pos[0])**2 + (self.YY_grid-pos[1])**2)
        dist[dist<0.2] = 0.2
        RSI = G0 / (4*np.pi*dist)**2
        return RSI

    def eval_J_minSIR(self,pos_est,pow_est,source_idxs):
        N_source_tot = len(pow_est)
        N_source = len(source_idxs)
        interferer_idxs = [x for x in range(N_source_tot) if not(x in source_idxs)]
        N_interferer = len(interferer_idxs)

        RSI_int = np.zeros((self.N_points[1],self.N_points[0]))
        for ii in range(N_interferer):
            RSI_tmp = self.eval_RSI(pos_est[interferer_idxs[ii],:],pow_est[interferer_idxs[ii]])
            RSI_int = RSI_int + RSI_tmp

        RSI_source = np.zeros((self.N_points[1],self.N_points[0],N_source))
        SIR_source = np.zeros((self.N_points[1],self.N_points[0],N_source))
        for ii in range(N_source):
            RSI_tmp = self.eval_RSI(pos_est[source_idxs[ii],:],pow_est[source_idxs[ii]])
            #print RSI_tmp.shape, type(RSI_tmp)
            RSI_source[:,:,ii] = RSI_tmp
            if N_interferer > 0:
                SIR_source[:,:,ii] = 10.*np.log10(RSI_tmp / RSI_int)
            else:
                SIR_source[:,:,ii] = 10.*np.log10(RSI_tmp)

        SIR_source[np.isnan(SIR_source)] = 30
        SIR_source[np.isinf(SIR_source)] = 30
        #rospy.loginfo(str(SIR_source))
        if self.LOG_DIR:
            SIR_source[SIR_source<-30] = -30
            SIR_source[SIR_source>30] = 30
            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            surf_tmp = ax1.imshow(np.flipud(np.min(SIR_source,axis=2)),cmap=plt.cm.gray,extent=[self.X_grid[0],self.X_grid[-1],self.Y_grid[0],self.Y_grid[-1]])
            divider = make_axes_locatable(ax1)
            cax = divider.append_axes("right", size="5%", pad=0.05)            
            self.fig.colorbar(surf_tmp,cax=cax)
            for kk in range(N_interferer):
                ax1.plot(pos_est[interferer_idxs[ii],0], pos_est[interferer_idxs[ii],1], 'o', ms=MS, color='r')

            for kk in range(N_source):
                ax1.plot(pos_est[source_idxs[ii],0], pos_est[source_idxs[ii],1], 'o', ms=MS, color='g')


            save_name = os.path.join(self.LOG_DIR,'J_SIR.png')
            ax1.tick_params(labelsize=FONTSIZE)
            ax1.set_xlabel('X coordinate',fontsize=FONTSIZE)
            ax1.set_ylabel('Y coordinate',fontsize=FONTSIZE)
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')
            plt.close(self.fig)

        return np.min(SIR_source,axis=2)


    def eval_constraint_dist(self,p):
        dist_mat = np.sqrt((p[0]-self.XX_grid)**2 + (p[1]-self.YY_grid)**2)
        dist_mat[dist_mat < self.j_soc_dist_param[0]] = -np.inf
        dist_mat[dist_mat > self.j_soc_dist_param[1]] = -np.inf
        dist_mat[dist_mat > 0] = 0

        return dist_mat

    def eval_constraint_angle(self,p,p_theta):
        angle_mat = np.arctan2(self.YY_grid-p[1],self.XX_grid-p[0])*180./np.pi
        idx_pos = angle_mat > 0
        angle_mat[idx_pos] = -360. + angle_mat[idx_pos]
        angle_mat[angle_mat < (self.j_soc_angle_param[0]+p_theta)] = -np.inf
        angle_mat[angle_mat > (self.j_soc_angle_param[1]+p_theta)] = -np.inf
        angle_mat[np.logical_not(np.isinf(angle_mat))] = 0

        return angle_mat


    def eval_constraint_social(self,P,human_idx,Theta):
        N_human = len(human_idx)

        J_mat = np.zeros((self.N_points[1],self.N_points[0]))

        for ii in range(N_human):
            J_mat = J_mat + self.eval_constraint_dist(P[human_idx[ii],:])
            J_mat = J_mat + self.eval_constraint_angle(P[human_idx[ii],:],Theta[ii])

        return J_mat


    def handle_angle_wrap(self,angle):
        # Function for wrapping angles to (-180;180]
        angle = angle % 360.
        angle = (angle+360.) % 360.
        #if angle > 180.:
        #    angle -= 360.

        angle[np.where(angle>180.)] -= 360  
        return angle


class AudioRecording:
    def __init__(self,device_array,device_single,data_path):
        rospy.loginfo('Waiting for audio service')
        rospy.wait_for_service('/Robot2/get_audio_segment_duo')
        rospy.loginfo('Done')
        self.device_array = device_array
        self.device_single = device_single
        self.data_path = data_path
        self.get_audio_segment_duo1 = rospy.ServiceProxy('/Robot2/get_audio_segment_duo', GetAudioSegmentDuo)
        self.count = 0

    def record_audio(self,N_sec):
        file_name1 = os.path.join(self.data_path,'audio_array_'+str(self.count)+'.wav')
        file_name2 = os.path.join(self.data_path,'audio_single_'+str(self.count)+'.wav')
        command1 = 'arecord -D '+self.device_array+' -f cd -r 16000 -c 4 -d '+str(N_sec)+' '+file_name1
        command2 = 'arecord -D '+self.device_single+' -f cd -r 16000 -d '+str(N_sec)+' '+file_name2
        #rospy.loginfo('Recording Audio')
        resp1 = self.get_audio_segment_duo1(command1,file_name1,self.count,self.data_path,command2,file_name2,self.count,self.data_path)
        X1 = np.vstack((resp1.m1_ch1,resp1.m1_ch2,resp1.m1_ch3,resp1.m1_ch4)).T
        X2 = np.asarray(resp1.m2_ch1)
        #print X2.shape
        self.count += 1
        #return X/(2.**15)
        return X1,X2

class SourceClassificationLaser:
    def __init__(self,Laser_node,Room_dim,X_grid,Y_grid,LOG_DIR):
        self.t0 = rospy.get_param('/T0')
        self.LOG_DIR = LOG_DIR    
        self.LOG_FILE = os.path.join(self.LOG_DIR,'SourceClassification.txt') 

        self.Room_dim = Room_dim
        self.X_grid = X_grid
        self.Y_grid = Y_grid
        self.Nx = len(self.X_grid)
        self.Ny = len(self.Y_grid)
        self.Tau_std = 10.
        self.XX_grid,self.YY_grid = np.meshgrid(self.X_grid,self.Y_grid)
        self.log_count = 0
        self.N_scans = 5
        self.Laser_node = Laser_node
        self.Laser_angle_array = np.zeros((0,10)) 
        self.MIN_DIST = 1.

    def log_leg_detection(self,x):
        M,N = x.shape
        t_sec = rospy.get_time()-self.t0
        f = open(self.LOG_FILE,'a')
        for mm in range(M):
            x1 = np.concatenate((np.array(t_sec).reshape(1,1),x[mm,:].reshape(1,10)),axis=1)
            np.savetxt(f,x1,delimiter=' ',fmt='%4.2f')
        f.close()


    def detect_human(self,robot_pos,robot_pose):
        Laser_angle_array_tmp = np.zeros((0,10))
        for ii in range(self.N_scans):
            leg_msg_tmp = rospy.wait_for_message(self.Laser_node,multiple_legs,timeout=1)
            if leg_msg_tmp.N_leg_pairs > 0:
                for kk in range(leg_msg_tmp.N_leg_pairs):
                    rospy.loginfo('Leg Pair Detected: '+str(leg_msg_tmp.leg_pairs[kk].angle))
                    Laser_angle_array_tmp = np.concatenate((Laser_angle_array_tmp,np.array([self.log_count,robot_pos[0],robot_pos[1],robot_pose,leg_msg_tmp.leg_pairs[kk].angle,leg_msg_tmp.leg_pairs[kk].range,leg_msg_tmp.leg_pairs[kk].angle_1,leg_msg_tmp.leg_pairs[kk].range_1,leg_msg_tmp.leg_pairs[kk].angle_2,leg_msg_tmp.leg_pairs[kk].range_2]).reshape(1,10)),axis=0)               
            else:
                rospy.loginfo('No Legs Detected')
            rospy.sleep(0.2)
        self.log_count += 1
        self.Laser_angle_array = np.concatenate((self.Laser_angle_array,Laser_angle_array_tmp),axis=0)
        self.log_leg_detection(Laser_angle_array_tmp)
        #rospy.loginfo(str(self.Laser_angle_array))


    def pos_estimation(self):
        N_meas = self.Laser_angle_array.shape[0]
        #rospy.loginfo(str(self.Laser_angle_array))
        rospy.loginfo('N meas: '+str(self.Laser_angle_array.shape))
        person_clusters = np.zeros((0,6))
        if N_meas > 0:
            rob_pos_init = self.Laser_angle_array[0,1:4]
            for kk in range(N_meas):
                person_angle = -self.Laser_angle_array[kk,4] #mirrored according to this convention
                person_range = self.Laser_angle_array[kk,5] 
                robot_odom = self.Laser_angle_array[kk,1:4]
                rel_pos_person = np.zeros(2,)
                rel_pos_person[0] = person_range*np.sin((person_angle+robot_odom[2])*np.pi/180.)
                rel_pos_person[1] = person_range*np.cos((person_angle+robot_odom[2])*np.pi/180.)
                #abs_pos_person = np.array([0,0])
                #abs_pos_person[0] = robot_odom[0] + person_range*np.sin((person_angle+robot_odom[2])*np.pi/180.)
                #abs_pos_person[1] = robot_odom[1] + person_range*np.cos((person_angle+robot_odom[2])*np.pi/180.)
                abs_pos_person = robot_odom[0:2]+rel_pos_person
                rospy.loginfo('----')
                rospy.loginfo('Person: '+str(person_range)+' '+str(person_angle)+' '+str(person_range*np.sin((person_angle)*np.pi/180.)))
                rospy.loginfo('Abs pos: '+str(abs_pos_person)+' '+str(robot_odom)+' '+str(rel_pos_person))
                if abs_pos_person[0] > 0 and abs_pos_person[0] <= self.Room_dim[0] and abs_pos_person[1] > 0 and abs_pos_person[1] <= self.Room_dim[1]: #To make sure the person is wihtin the valid area
                    rospy.loginfo('#kk = '+str(kk))
                    if person_clusters.shape[0] == 0:
                        tmp_person = np.zeros(6,)
                        tmp_person[0:2] = abs_pos_person.reshape(2,)
                        tmp_person[2] = robot_odom[0] + self.Laser_angle_array[kk,7]*np.sin((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                        tmp_person[3] = robot_odom[1] + self.Laser_angle_array[kk,7]*np.cos((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                        tmp_person[4] = robot_odom[0] + self.Laser_angle_array[kk,9]*np.sin((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                        tmp_person[5] = robot_odom[1] + self.Laser_angle_array[kk,9]*np.cos((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                        person_clusters = np.concatenate((person_clusters,tmp_person.reshape(1,6)),axis=0)
                    else:
                        dist_vec = s_dist.cdist(person_clusters[:,0:2],np.tile(abs_pos_person.reshape(1,2),(person_clusters.shape[0],1)))[:,0]
                        if np.min(dist_vec) < self.MIN_DIST:
                            tmp_person = np.zeros(6,)
                            tmp_person[0:2] = abs_pos_person.reshape(2,)
                            tmp_person[2] = robot_odom[0] + self.Laser_angle_array[kk,7]*np.sin((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                            tmp_person[3] = robot_odom[1] + self.Laser_angle_array[kk,7]*np.cos((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                            tmp_person[4] = robot_odom[0] + self.Laser_angle_array[kk,9]*np.sin((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                            tmp_person[5] = robot_odom[1] + self.Laser_angle_array[kk,9]*np.cos((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                            min_idx = np.argmin(dist_vec)
                            person_clusters[min_idx,:] = (person_clusters[min_idx,:] + tmp_person)/2. #Bad implementation -> bias towards later estimates
                        else:
                            tmp_person = np.zeros(6,)
                            tmp_person[0:2] = abs_pos_person.reshape(2,)
                            tmp_person[2] = robot_odom[0] + self.Laser_angle_array[kk,7]*np.sin((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                            tmp_person[3] = robot_odom[1] + self.Laser_angle_array[kk,7]*np.cos((-self.Laser_angle_array[kk,6]+robot_odom[2])*np.pi/180.)
                            tmp_person[4] = robot_odom[0] + self.Laser_angle_array[kk,9]*np.sin((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                            tmp_person[5] = robot_odom[1] + self.Laser_angle_array[kk,9]*np.cos((-self.Laser_angle_array[kk,8]+robot_odom[2])*np.pi/180.)
                            person_clusters = np.concatenate((person_clusters,tmp_person.reshape(1,6)),axis=0)   

            N_clusters = person_clusters.shape[0]
            pose_vec = np.zeros(N_clusters,)
            rospy.loginfo('N_cluster:::::: '+str(N_clusters))
            for dd in range(N_clusters):
                angle1 = np.arctan2(person_clusters[dd,3]-rob_pos_init[1],person_clusters[dd,2]-rob_pos_init[0])
                angle2 = np.arctan2(person_clusters[dd,5]-rob_pos_init[1],person_clusters[dd,4]-rob_pos_init[0])
                rospy.loginfo('Angle 1: '+str(angle1)+' Angle 2: '+str(angle2))
                if angle1 > angle2:
                    pose_vec[dd] = np.arctan2(person_clusters[dd,5]-person_clusters[dd,3],person_clusters[dd,4]-person_clusters[dd,2])*180./np.pi - 90.
                else:
                    pose_vec[dd] = np.arctan2(person_clusters[dd,3]-person_clusters[dd,5],person_clusters[dd,2]-person_clusters[dd,4])*180./np.pi - 90.

                init_angle_tmp = np.arctan2(rob_pos_init[1]-person_clusters[dd,1],rob_pos_init[0]-person_clusters[dd,0])*180./np.pi
                rospy.loginfo('Init_angle: '+str(init_angle_tmp))
                if pose_vec[dd] < (-90.+init_angle_tmp):
                    pose_vec[dd] += 180.
                if pose_vec[dd] > (90.+init_angle_tmp):
                    pose_vec[dd] -= 180.

            if self.LOG_DIR:
                self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
                ax1 = self.fig.add_subplot(111)
                for mm in range(N_clusters):
                    ax1.plot(person_clusters[mm,0], person_clusters[mm,1], 'o', ms=MS*2, color='r') #Person
                    ax1.plot(person_clusters[mm,2], person_clusters[mm,3], '+', ms=MS*2, color='b') #Person
                    ax1.plot(person_clusters[mm,4], person_clusters[mm,5], '+', ms=MS*2, color='b') #Person 
                    ax1.arrow(person_clusters[mm,0], person_clusters[mm,1], np.cos(pose_vec[mm]*np.pi/180.)*ARROW_LEN,np.sin(pose_vec[mm]*np.pi/180.)*ARROW_LEN, head_width=0.05, head_length=0.1, fc='g', ec='g')


                save_name = os.path.join(self.LOG_DIR,'Leg_detection.png')
                ax1.set_xlim([self.X_grid[0], self.X_grid[-1]])
                ax1.set_ylim([self.Y_grid[0], self.Y_grid[-1]])
                ax1.tick_params(labelsize=FONTSIZE)
                ax1.set_xlabel('X coordinate',fontsize=FONTSIZE)
                ax1.set_ylabel('Y coordinate',fontsize=FONTSIZE)
                self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')
                plt.close(self.fig)

                f_name = os.path.join(self.LOG_DIR,'Leg_detection.h5')
                h5f = h5py.File(f_name, 'w')
                h5f.create_dataset('person_clusters', data=person_clusters)
                h5f.create_dataset('person_pose', data=pose_vec)
                h5f.close()


        else:
            person_clusters = np.array([])
            pose_vec = np.array([])

        #rospy.loginfo('N_pose:::::: '+str(pose_vec.shape))
        return person_clusters,pose_vec

    def match_source(self,speech_position_estimates):
        laser_pos_est,laser_pose_est = self.pos_estimation()
        if laser_pos_est.size>0:
            rospy.loginfo('Match Sources')
            rospy.loginfo(str(laser_pos_est))
            rospy.loginfo(str(laser_pos_est.shape))
            rospy.loginfo(str(speech_position_estimates))
            rospy.loginfo(str(speech_position_estimates.shape))
            N_humans = laser_pos_est.shape[0]
            idx_list = []

            for ii in range(N_humans):
                dist_vec = s_dist.cdist(speech_position_estimates,np.tile(laser_pos_est[ii,0:2].reshape(1,2),(speech_position_estimates.shape[0],1)))[:,0]
                min_idx = np.argmin(dist_vec)
                if dist_vec[min_idx] <= self.MIN_DIST:
                    idx_list.append(min_idx)
                    np.delete(speech_position_estimates,min_idx,0)

            rospy.loginfo(str(idx_list))
            return idx_list,laser_pose_est[idx_list]
        else:
            return [],np.array([])


class PosEstimation:
    def __init__(self,X_grid,Y_grid,noise_std,LOG_DIR):
        self.t0 = rospy.get_param('/T0')
        self.LOG_DIR = LOG_DIR    
        self.LOG_FILE = os.path.join(self.LOG_DIR,'POS.txt')
            
        self.X_grid = X_grid
        self.Y_grid = Y_grid
        self.Nx = len(self.X_grid)
        self.Ny = len(self.Y_grid)
        self.Tau_std = noise_std
        self.XX_grid,self.YY_grid = np.meshgrid(self.X_grid,self.Y_grid)

        self.log_count = 0


    def log_pos(self,x):
        M,N = x.shape
        t_sec = rospy.get_time()-self.t0
        f = open(self.LOG_FILE,'a')
        x1 = np.hstack((np.array(t_sec).reshape(1,1),x.reshape(1,M*N)))
        np.savetxt(f,x1,delimiter=' ',fmt='%4.2f')
        f.close()

    def handle_angle_wrap(self,angle):
        # Function for wrapping angles to (-180;180]
        angle = angle % 360.
        angle = (angle+360.) % 360.
        idx = angle>180.
        angle[idx] = angle[idx]-360

        return angle


    def mask_angle(self,p,p_theta,angle_thres):
        angle_mat = np.arctan2(self.YY_grid-p[1],self.XX_grid-p[0])*180./np.pi - 90. + p_theta
        angle_mat = self.handle_angle_wrap(angle_mat)
        mask_mat = np.ones(angle_mat.shape)
        mask_mat[angle_mat < angle_thres[0]] = 0
        mask_mat[angle_mat > angle_thres[1]] = 0

        return mask_mat

    def gridMlMs(self,Tau_meas,Robot_pos,Robot_phi,Array_geo):
        N_meas,N_sources = Tau_meas.shape
        mask_mat = self.mask_angle(Robot_pos[-1,0:2],Robot_phi[-1],np.array([-90.,90.]))
        #mask_mat = np.flipud(mask_mat)
        #rospy.loginfo('N_sources '+str(N_sources))
        #rospy.loginfo('N_meas '+str(N_meas))
        J_mat_2 = np.zeros((self.Ny,self.Nx,N_sources))
        for nn in range(N_sources):
            J_mat_tmp = np.ones((self.Ny,self.Nx))
            for mm in range(N_meas):
                M1 = Robot_pos[mm,0:2]+rotate_point(Array_geo[0,0:2],Robot_phi[mm])
                M2 = Robot_pos[mm,0:2]+rotate_point(Array_geo[1,0:2],Robot_phi[mm])
                Tau_hyp = self.hypTdoa(self.XX_grid,self.YY_grid,M1,M2)
                
                J_mat_tmp = np.multiply(J_mat_tmp, self.normPdf(Tau_meas[mm,nn],Tau_hyp,self.Tau_std)) 
                J_mat_tmp = np.multiply(J_mat_tmp,mask_mat)
            J_mat_2[:,:,nn] = J_mat_tmp / np.sum(np.sum(J_mat_tmp))

        

        pos_mat_2 = np.zeros((N_sources,2))


        for dd in range(N_sources):
            J_mat2 = self.findPeak2D(np.squeeze(J_mat_2[:,:,dd]),5)
            pos_idx1 = np.unravel_index(J_mat2.flatten().argsort()[::-1][0],J_mat2.shape)
            pos_mat_2[dd,0] = self.X_grid[pos_idx1[1]]
            pos_mat_2[dd,1] = self.Y_grid[pos_idx1[0]]


        if self.LOG_DIR:
            self.log_pos(pos_mat_2)
            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            surf_tmp = ax1.imshow(np.flipud(np.mean(J_mat_2,axis=2)),cmap=plt.cm.gray,extent=[self.X_grid[0],self.X_grid[-1],self.Y_grid[0],self.Y_grid[-1]])
            divider = make_axes_locatable(ax1)
            cax = divider.append_axes("right", size="5%", pad=0.05)            
            self.fig.colorbar(surf_tmp,cax=cax)
            for kk in range(N_meas):
                ax1.plot(Robot_pos[kk,0], Robot_pos[kk,1], 'o', ms=MS, color='r')
                ax1.arrow(Robot_pos[kk,0], Robot_pos[kk,1], np.sin(Robot_phi[kk])*ARROW_LEN,np.cos(Robot_phi[kk])*ARROW_LEN, head_width=0.05, head_length=0.1, fc='g', ec='g')
            
            for kk in range(N_sources):
                ax1.plot(pos_mat_2[kk,0], pos_mat_2[kk,1], 'o', ms=MS, color='g')


            f_name = os.path.join(self.LOG_DIR,'POS_ML_'+str(self.log_count)+'.h5')
            h5f = h5py.File(f_name, 'w')
            h5f.create_dataset('likelihood_mat', data=J_mat_2)
            h5f.create_dataset('position_estimates', data=pos_mat_2)
            h5f.create_dataset('robot_traj', data=Robot_pos)
            h5f.close()


            save_name = os.path.join(self.LOG_DIR,'POS_ML_'+str(self.log_count)+'.png')
            ax1.tick_params(labelsize=FONTSIZE)
            ax1.set_xlabel('X coordinate',fontsize=FONTSIZE)
            ax1.set_ylabel('Y coordinate',fontsize=FONTSIZE)
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')
            plt.close(self.fig)

        if self.log_count == 0:
            self.pos_est_ar = pos_mat_2.reshape(N_sources,2,1)
        else:
            self.pos_est_ar = np.append(self.pos_est_ar,pos_mat_2.reshape(N_sources,2,1),axis=2)
        
        self.log_count += 1   

        return J_mat_2,J_mat_2,pos_mat_2


    def hypTdoa(self,x,y,M1,M2):
        return (np.sqrt((x - M1[0])**2 + (y - M1[1])**2) - np.sqrt((x - M2[0])**2 +(y - M2[1])**2) ) / C
        
    def normPdf(self,x,mu,sigma):
        return (1. / np.sqrt(2.*np.pi*sigma**2)) * np.exp(-(x-mu)**2 / (2.*sigma**2))


    def findPeak2D(self,X,N_point):
        M,N = X.shape
        Y = np.zeros((M,N))

        for mm in range(M):
            idx_m1 = np.max([0,mm-N_point])
            idx_m2 = np.min([M-1,mm+N_point])
            for nn in range(N):
                idx_n1 = np.max([0,nn-N_point])
                idx_n2 = np.min([N-1,nn+N_point])               
                local_X = X[idx_m1:idx_m2+1,idx_n1:idx_n2+1]
                if np.sum((local_X>=X[mm,nn]).flatten()) <= 1:
                    Y[mm,nn] = X[mm,nn]
    
        return Y


class TDOA:
    def __init__(self,mic_array_geo,N_sources,log_dir):
        self.t0 = rospy.get_param('/T0')
        self.LOG_DIR = log_dir
        self.LOG_FILE = os.path.join(self.LOG_DIR,'log_tdoa.txt')
        self.N_win = 1024
        self.N_overlap = round(self.N_win*0.75)
        self.N_advance = self.N_win - self.N_overlap
        self.N_fft = 2**10
        self.Omega_size = 64
        self.Omega_overlap = round(self.Omega_size*0.75)
        self.Omega_advance = self.Omega_size - self.Omega_overlap
        self.Omega_eps = 0.2
        self.Omega_Nwin = 10
        self.E_thres = -70 #-70#%10*log10(1e-7);
        self.N_bins = 100
        self.N_sources = N_sources
        self.do_filter = 1
        self.N_point_peak = 2
        self.Peak_thres = 0.1
        self.mic_dist = np.linalg.norm(mic_array_geo[0,:]-mic_array_geo[1,:])
        self.max_delay = self.mic_dist/C
        self.delay_vec = np.linspace(-self.max_delay,self.max_delay,self.N_bins)*16000.
        
        self.count = 0


    def log_tdoa(self,x):
        t_sec = rospy.get_time()-self.t0
        f = open(self.LOG_FILE,'a')
        x1 = np.hstack((np.array(t_sec).reshape(1,1),np.array(self.count).reshape(1,1),x.reshape(1,x.shape[0])))
        np.savetxt(f,x1,delimiter=' ',fmt='%4.3e')
        f.close()


    def estimate_tdoa(self,X):
        e_count = 0
        win = np.hanning(self.N_win)
        N_samples = X.shape[0]
        N_frames = np.floor((N_samples-self.N_win)/self.N_advance) + 1
        e_arr = np.zeros((N_frames,)) 
        delay_est = []
        for ii in range(int(N_frames)): #Process signal in frames
            #print "Iteration: ", ii+1, "/", int(N_frames)
            if ii==0:
                idx_start = 0
                idx_stop = self.N_win
            else:
                idx_start = idx_start+self.N_advance
                idx_stop = idx_start+self.N_win

            #print idx_start, idx_stop
            x1_temp = X[idx_start:idx_stop,3]
            x2_temp = X[idx_start:idx_stop,0]

            X1_temp = np.fft.fft(x1_temp*win,self.N_fft)
            X2_temp = np.fft.fft(x2_temp*win,self.N_fft)

            #print 'NaN: ', np.sum(np.isnan(X1_temp)), X1_temp.shape
            energy_tmp = 10*np.log10(np.mean(x1_temp**2))
            e_arr[ii] = energy_tmp
            rospy.loginfo
            if energy_tmp > self.E_thres: #Simple Energy VAD
                e_count += 1
                idx_list = self.getCorr(X1_temp,X2_temp) #Find TF zones
                N_tf = len(idx_list)
                #print '#TF: ', N_tf
                if N_tf>0: #Only process further if at least one zone is found
                    X_cross = np.multiply(X1_temp,X2_temp.conjugate())
                    X_cross_abs = np.abs(X_cross)
                    X_12 = np.divide(X_cross,X_cross_abs)
                    for ff in range(N_tf): #Loop over TF zones and do time-delay estimation on those zones
                        idx_vec = np.arange(idx_list[ff][0],idx_list[ff][1])
                        I_vec,D_vec = np.meshgrid(idx_vec,self.delay_vec)
                        ifft_mat = np.exp(2j*np.pi*I_vec*D_vec/self.N_fft)
                        x_cross = np.real(np.dot(ifft_mat,X_12[idx_list[ff][0]:idx_list[ff][1]]))
                        max_idx = np.argmax(x_cross)
                        delay_est.append(self.delay_vec[max_idx]/16000.)
                        
        #print 'Energy VAD: ', e_count, '/', N_frames           
        delay_est = np.asarray(delay_est)
        counts,edges = np.histogram(delay_est,self.delay_vec/16000)
        counts[0:2] = np.min(counts)
        counts[-2:] = np.min(counts)         
        if self.do_filter:
                counts = signal.filtfilt(np.ones(3,)/3.,1,counts)        
        counts = counts / np.sum(counts).astype(float)
        max_idx = self.findLocalMaxima(counts)
        tau_hat = edges[max_idx]
        tau_hat.sort() #!!!Dirty hack to make sure the delays are associated with the right source. Will only work when there is no rotation in array
        edges = edges[0:-1] + np.diff(edges)/2.

        if self.LOG_FILE:
            self.log_tdoa(tau_hat)
            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            ax1.bar(edges,counts, align='center', width=np.abs(edges[1]-edges[0]))
            ax1.set_xlabel('TDOA [s]',fontsize=FONTSIZE)
            ax1.set_ylabel('Pr',fontsize=FONTSIZE)
            ax1.tick_params(labelsize=FONTSIZE)
            save_name = os.path.join(self.LOG_DIR,'TDOA_histogram_'+str(self.count)+'.png')
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')            
            plt.close(self.fig) 

            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            ax1.plot(e_arr)
            save_name = os.path.join(self.LOG_DIR,'E_arr_'+str(self.count)+'.png')
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')            
            plt.close(self.fig)    


        if self.count == 0:
            self.Tau_ar = tau_hat.reshape(1,self.N_sources)
        else:
            self.Tau_ar = np.vstack((self.Tau_ar,tau_hat.reshape(1,self.N_sources)))      
          
        self.count += 1    
        return counts,edges,tau_hat


    def getCorr(self,X1,X2):
        idx_list = []
        for ii in range(int(self.Omega_Nwin)):
            #print "Iteration: ", ii+1 #"/", int(N_WIN)
            if ii==0:
                idx_start = 0
                idx_stop = self.Omega_size
            else:
                idx_start = idx_start+self.Omega_advance
                idx_stop = idx_start+self.Omega_size

            #print idx_start, idx_stop, X1.shape
            X1_temp = X1[idx_start:idx_stop]
            X2_temp = X2[idx_start:idx_stop]

            r_num = np.sum(np.abs(X1_temp*X2_temp))
            r_den1 = np.sum(np.abs(X1_temp*X1_temp))    
            r_den2 = np.sum(np.abs(X2_temp*X2_temp))

            rr = r_num / np.sqrt(r_den1*r_den2) 
            #print 'rr: ', rr, r_num, r_den1, r_den2
            if rr>(1-self.Omega_eps) and ii>0:
                idx_list.append([idx_start,idx_stop])

        return idx_list


    def findLocalMaxima(self,x):
        N_samples = len(x)
        y = np.zeros(N_samples)

        for dd in range(N_samples):
            idx1 = np.max([0, dd-self.N_point_peak])
            idx2 = np.min([N_samples-1,dd+self.N_point_peak+1])
            if dd < (N_samples-1): #Dirty hack to make sure that adjacent bins with exact equal value/count does not equal two peaks 
                if x[dd] == x[dd+1]:
                    x[dd+1] = x[dd+1]*0.999
            if np.sum(x[idx1:idx2]>x[dd]) == 0:
                y[dd] = x[dd]

        #y = y / np.sum(y).astype(float)
        #y[y<self.Peak_thres] = 0
        max_idx = y.argsort()[::-1][:self.N_sources]
        
        
        return max_idx


class RobotMove:
    def __init__(self,init_pos,step_size,mic_array,room_size,move_topic,odom_topic,move_base_topic,log_dir):
        self.t0 = rospy.get_param('/T0') #rospy.get_time()
        self.init_pos = init_pos
        self.move_topic = move_topic
        self.odom_topic = odom_topic
        self.move_pub = rospy.Publisher(self.move_topic,Twist,queue_size=1)
        self.angle_speed = 20.*np.pi/180. #Rotational speed
        self.linear_speed = 0.2 #0.2 0.15 #Speed when driving straight
        self.mic_array_geo = mic_array 
        self.room_size = room_size

        initial_pub = rospy.Publisher("/initialpose", PoseWithCovarianceStamped, queue_size=1)
        initial_msg = PoseWithCovarianceStamped()
        initial_msg.pose.pose.position.x = 0
        initial_msg.pose.pose.position.y = 0
        initial_msg.pose.pose.orientation.w = 1

        rospy.loginfo('Setting initial position')
        initial_pub.publish(initial_msg)
        rospy.loginfo('Done Setting initial position')

        rospy.loginfo('Setting up server')
        self.sac = actionlib.SimpleActionClient(move_base_topic, MoveBaseAction)
        self.sac.wait_for_server()
        rospy.loginfo('Done Setting up server')

        self.N_mic = self.mic_array_geo.shape[0]
        self.crlb_grid_res = 0.1
        self.crlb_grid_range = np.floor(step_size / self.crlb_grid_res) * self.crlb_grid_res
        self.crlb_x_grid = np.linspace(-self.crlb_grid_range,self.crlb_grid_range,2*self.crlb_grid_range/self.crlb_grid_res+1)
        self.crlb_y_grid = np.linspace(0,self.crlb_grid_range,self.crlb_grid_range/self.crlb_grid_res+1)
        self.Nx = len(self.crlb_x_grid)
        self.Ny = len(self.crlb_y_grid)

        self.LOG_DIR = log_dir
        self.log_file_nav = os.path.join(self.LOG_DIR,'Robot_odom.txt')
        self.Fid_nav = open(self.log_file_nav,'a')
        self.legend_nav = ['Time','X','Y','Orientation']
        self.Fid_nav.write(" ".join(self.legend_nav))
        self.Fid_nav.write("\n")
        self.Fid_nav.close()

        self.log_file_opt = os.path.join(self.LOG_DIR,'Robot_opt_move.txt')
        self.Fid_opt = open(self.log_file_opt,'a')
        self.legend_nav = ['Time','X','Y','Orientation']
        self.Fid_opt.write(" ".join(self.legend_nav))
        self.Fid_opt.write("\n")
        self.Fid_opt.close()

        self.log_count = 0
        self.seq_count = 0

        self.bumper_flag = False
        rospy.Subscriber('/mobile_base/events/bumper',BumperEvent,self.cb_bumper)



    def move_pos_abs_pos(self,pos):
        if pos[0] > 0 and pos[0] < self.room_size[0] and pos[1] > 0 and pos[1] < self.room_size[1]:
            cur_pos = self.get_state()
            move_dist = np.linalg.norm(pos-cur_pos[0:2])
            move_angle = np.arctan2(pos[0]-cur_pos[0],pos[1]-cur_pos[1])*180./np.pi - cur_pos[2]
            #self.make_move1(move_angle,0.)
            rospy.sleep(0.2)
            goal_msg = MoveBaseGoal()
            goal_msg.target_pose.header.seq = self.seq_count
            goal_msg.target_pose.header.stamp = rospy.Time.now()
            goal_msg.target_pose.header.frame_id = '/odom'    
            goal_msg.target_pose.pose.position.x = pos[1]
            goal_msg.target_pose.pose.position.y = pos[0]
            goal_msg.target_pose.pose.orientation.w = 1         
            self.sac.send_goal(goal_msg)

            rospy.sleep(0.5)
            while True:
                ret_msg = rospy.wait_for_message('/move_base/status',GoalStatusArray, timeout=10)
                if ret_msg.status_list[0].status == 3:
                    rospy.loginfo('Goal Reached')
                    return True
                rospy.sleep(0.1)
        else:
            rospy.loginfo('Position outside room - doing nothing!')

    def move_pos_abs_pos1(self,position):
        if position[0] > 0 and position[0] < self.room_size[0] and position[1] > 0 and position[1] < self.room_size[1]:
            goal_msg = MoveBaseGoal()
            goal_msg.target_pose.header.seq = self.seq_count
            goal_msg.target_pose.header.stamp = rospy.Time.now()
            goal_msg.target_pose.header.frame_id = '/odom'    
            goal_msg.target_pose.pose.position.x = position[1] - self.init_pos[1]
            goal_msg.target_pose.pose.position.y = -1*(position[0] - self.init_pos[0])
            goal_msg.target_pose.pose.orientation.w = 1         
            self.sac.send_goal(goal_msg)
            self.sac.wait_for_result(rospy.Duration(0.))
            # rospy.sleep(0.5)
            # while True:
            #     ret_msg = rospy.wait_for_message('/move_base/status',GoalStatusArray, timeout=10)
            #     if ret_msg.status_list[0].status == 3:
            #         rospy.loginfo('Goal Reached')
            #         return True
            #     rospy.sleep(0.1)
            return True
        else:
            rospy.loginfo('Position outside room - doing nothing!')
            return False

    def cb_bumper(self,data):
        if data.state == 1:
            self.bumper_flag = True


    def write_log(self):
        angle = self.get_orientation()
        pos = self.get_position()
        t_sec = rospy.get_time()-self.t0
        w_vec = np.array([t_sec, pos[0], pos[1], angle]).reshape(1,4)
        self.Fid = open(self.log_file_nav,'a')
        np.savetxt(self.Fid,w_vec,delimiter=' ',fmt='%10.2f')
        self.Fid.close()

    def write_log_opt(self,new_move,new_phi):
        angle = new_phi
        pos = new_move
        t_sec = rospy.get_time()-self.t0
        w_vec = np.array([t_sec, pos[0], pos[1], angle]).reshape(1,4)
        self.Fid = open(self.log_file_opt,'a')
        np.savetxt(self.Fid,w_vec,delimiter=' ',fmt='%10.2f')
        self.Fid.close()

    def print_log(self):
        angle = self.get_orientation()
        pos = self.get_position()
        t_sec = rospy.get_time()-self.t0
        rospy.loginfo('Robot info: ')
        rospy.loginfo('Time: '+str(t_sec)+' - Pos, x: '+str(pos[1])+' - Pos, y: '+str(pos[0])+' - Heading: '+str(angle))


    def get_orientation(self):
        msg_angle = rospy.wait_for_message(self.odom_topic,Odometry, timeout=10) #Get odometry
        quat = (msg_angle.pose.pose.orientation.x, msg_angle.pose.pose.orientation.y, msg_angle.pose.pose.orientation.z, msg_angle.pose.pose.orientation.w)
        eul = euler_from_quaternion(quat)
        angle_off = eul[2]*180./np.pi
        return -angle_off

    def get_position(self):
        msg_angle = rospy.wait_for_message(self.odom_topic,Odometry, timeout=10) #Get odometry
        return -msg_angle.pose.pose.position.y+self.init_pos[0],msg_angle.pose.pose.position.x+self.init_pos[1]

    def get_state(self):
        state = np.zeros((3,))
        pos = self.get_position()
        phi = self.get_orientation()
        state[0] = pos[0]
        state[1] = pos[1]
        state[2] = phi

        return state


    def log_state(self):
        tmp_state = self.get_state()
        if self.log_count == 0:
            self.log_state_ar = tmp_state.reshape(1,3)
        else:
            self.log_state_ar = np.vstack((self.log_state_ar,tmp_state.reshape(1,3)))

        self.write_log()
        self.log_count += 1


    def rotate(self,angle):
        angle=-angle
        #Simple closed-loop controller for turning angle degrees
        init_angle = -self.get_orientation()
        target_angle = self.handle_angle_wrap(init_angle+angle)
        #abs_angle = np.abs(angle)
        #cur_angle = init_angle
        delta_angle = angle
        msg_twist = Twist()
        #rospy.loginfo('Init - delta_angle: '+str(delta_angle))

        while np.abs(delta_angle) > 1.5:
            #rospy.loginfo('Rotating - delta_angle: '+str(delta_angle))
            msg_twist.angular.z = np.sign(angle)*self.angle_speed
            self.move_pub.publish(msg_twist)
            delta_angle = self.handle_angle_wrap(target_angle-(-1)*self.get_orientation())

        #To stop it again
        for ii in range(5):
            msg_twist.angular.z = 0
            self.move_pub.publish(msg_twist)       

        return delta_angle


    def drive(self,dist):
        # Closed-loop controller for driving the robot straight. 
        init_pos = self.get_position()
        delta_dist = 0
        msg_twist = Twist()
        while delta_dist < np.abs(dist) and (not self.bumper_flag):
            #rospy.loginfo('Driving - delta_dist: '+str(delta_dist))
            #rospy.loginfo('Driving - Pos: '+str(self.get_position()))
            #rospy.loginfo('Driving - Orientation: '+str(self.get_orientation()))
            msg_twist.linear.x = np.sign(dist)*self.linear_speed
            self.move_pub.publish(msg_twist)
            cur_pos = self.get_position()
            delta_dist = np.sqrt((cur_pos[0]-init_pos[0])**2 + (cur_pos[1]-init_pos[1])**2)

        #To stop it again
        for ii in range(5):
            msg_twist.linear.x = 0
            self.move_pub.publish(msg_twist) 


        if self.bumper_flag: # If bumper was activated during driving, then the robot must back
            init_pos1 = self.get_position()
            target_dist = 0.6
            delta_dist1 = 0
            prev_pos = self.get_position()[1]
            while delta_dist1 < target_dist:
                #rospy.loginfo('Backing - delta_dist: '+str(delta_dist1))
                msg_twist.linear.x = -self.linear_speed*0.5
                self.move_pub.publish(msg_twist)
                cur_pos1 = self.get_position()
                delta_dist1 = np.sqrt((cur_pos1[0]-init_pos1[0])**2 + (cur_pos1[1]-init_pos1[1])**2)


        self.bumper_flag = False
        return np.abs(dist-delta_dist)

    def make_move(self,angle,distance):
        rospy.loginfo('Robot-Move: Angle: '+str(angle)+' - Distance: '+str(distance))
        self.rotate(angle)
        self.drive(distance)
        self.rotate(-angle) #This is done to return with same heading as before, which is necessary when checking size of gaps


    def make_move1(self,angle,distance):
        rospy.loginfo('Robot-Move 1: Angle: '+str(angle)+' - Distance: '+str(distance))
        self.rotate(angle)
        self.drive(distance)

    def make_move2(self,angle,distance,angle1):
        rospy.loginfo('Robot-Move 2: Angle1: '+str(angle)+' - Angle2: '+str(angle1)+' - Distance: '+str(distance))
        self.rotate(angle)
        self.drive(distance)
        self.rotate(angle1)

    def move2pos(self,pos,orientation):
        if pos[0] > 0 and pos[0] < self.room_size[0] and pos[1] > 0 and pos[1] < self.room_size[1]:
            cur_pos = self.get_state()
            move_dist = np.linalg.norm(pos-cur_pos[0:2])
            move_angle = np.arctan2(pos[0]-cur_pos[0],pos[1]-cur_pos[1])*180./np.pi - cur_pos[2]
            self.make_move2(move_angle,move_dist,-move_angle)
            new_orientation = self.handle_angle_wrap(orientation-cur_pos[2])
            self.make_move1(new_orientation,0.)
        else:
            rospy.loginfo('Position outside room - doing nothing!')

    def handle_angle_wrap(self,angle):
        # Function for wrapping angles to (-180;180]
        angle = angle % 360.
        angle = (angle+360.) % 360.
        if angle > 180.:
            angle -= 360.

        return angle

    def compute_crlb(self,Sx,Sy,Tau_var=1,c=1.,new_pos=None,new_phi=0,alpha_reg=0):
        S = np.array([Sx,Sy,1.5])
        N_pos = self.log_state_ar.shape[0]
        
        Fisher_mat = np.zeros((2,2))
        for nn in range(N_pos):
            q1 = self.log_state_ar[nn,:] + rotate_point(self.mic_array_geo[0,:],self.log_state_ar[nn,2])
            q2 = self.log_state_ar[nn,:] + rotate_point(self.mic_array_geo[1,:],self.log_state_ar[nn,2])
            norm_q1 = np.linalg.norm(q1-S.reshape(3,))
            norm_q2 = np.linalg.norm(q2-S.reshape(3,))
            Fisher_mat[0,0] = Fisher_mat[0,0] + ((q1[0]-S[0])/norm_q1 - (q2[0]-S[0])/norm_q2)**2
            Fisher_mat[1,1] = Fisher_mat[1,1] + ((q1[1]-S[1])/norm_q1 - (q2[1]-S[1])/norm_q2)**2
            Fisher_mat[1,0] = Fisher_mat[1,0] + ((q1[0]-S[0])/norm_q1 - (q2[0]-S[0])/norm_q2) * ((q1[1]-S[1])/norm_q1 - (q2[1]-S[1])/norm_q2)


        if new_pos.any():
            q1 = new_pos.reshape(3,) + rotate_point(self.mic_array_geo[0,:],new_phi)
            q2 = new_pos.reshape(3,) + rotate_point(self.mic_array_geo[1,:],new_phi)
            norm_q1 = np.linalg.norm(q1-S.reshape(3,))
            norm_q2 = np.linalg.norm(q2-S.reshape(3,))
            Fisher_mat[0,0] = Fisher_mat[0,0] + ((q1[0]-S[0])/norm_q1 - (q2[0]-S[0])/norm_q2)**2
            Fisher_mat[1,1] = Fisher_mat[1,1] + ((q1[1]-S[1])/norm_q1 - (q2[1]-S[1])/norm_q2)**2
            Fisher_mat[1,0] = Fisher_mat[1,0] + ((q1[0]-S[0])/norm_q1 - (q2[0]-S[0])/norm_q2) * ((q1[1]-S[1])/norm_q1 - (q2[1]-S[1])/norm_q2)


        Fisher_mat[0,1] = Fisher_mat[1,0]
        Fisher_mat = Fisher_mat / (Tau_var* c**2)
        F_det = np.linalg.det(Fisher_mat) + alpha_reg
        crlb = np.sum(np.flipud(np.diag(Fisher_mat/F_det)))
        return crlb

    def compute_crlb_grid_weight(self,X_grid,Y_grid,W_mat,W_thres,Tau_var=1,new_pos=None,new_phi=None):
        Nx = len(X_grid)
        Ny = len(Y_grid)        
        crlb_grid = np.zeros((Ny,Nx))

        for xx in range(Nx):
            for yy in range(Ny):
                if W_mat[yy,xx] >= W_thres:
                    crlb_grid[yy,xx] = self.compute_crlb(X_grid[xx],Y_grid[yy],Tau_var,new_pos=new_pos,new_phi=new_phi,alpha_reg=0.0000001) * W_mat[yy,xx]

        return crlb_grid

    def find_opt_move(self,X_grid,Y_grid,W_mat,Pos_Est,Tau_var=1):
        N_sources = W_mat.shape[2]       

        J_mat = np.zeros((self.Ny,self.Nx))

        MAX_VAL = -1
        for yy in range(self.Ny):
            for xx in range(self.Nx):
                temp_move = np.array([self.crlb_x_grid[xx],self.crlb_y_grid[yy],0]).reshape(3,)
                temp_move = rotate_point(temp_move,self.log_state_ar[-1,2]*np.pi/180.)
                temp_phi = calc_orientation(self.log_state_ar[-1,:]+temp_move,Pos_Est)
                if self.validate_move(temp_move,Pos_Est):
                    for ss in range(N_sources):
                        W_sort = np.sort(W_mat[:,:,ss].flatten())[::-1]
                        W_cumsum = np.cumsum(W_sort)
                        Thres_val = W_sort[np.min([25,np.abs(W_cumsum-0.5).argsort()[0]])]  #20                     
                        crlb_tmp = self.compute_crlb_grid_weight(X_grid,Y_grid,np.squeeze(W_mat[:,:,ss]),Thres_val,new_pos=self.log_state_ar[-1,:]+temp_move,new_phi=temp_phi)
                        J_mat[yy,xx] = J_mat[yy,xx] + np.sum(np.sum(crlb_tmp))
                else:
                    J_mat[yy,xx] = MAX_VAL

        #Only for plotting purposes
        MAX_IDX = J_mat==MAX_VAL
        J_mat[MAX_IDX] = np.max(J_mat[J_mat!=MAX_VAL])

        if self.LOG_DIR:
            self.fig = plt.figure(figsize=FIG_SIZE,tight_layout=True)
            ax1 = self.fig.add_subplot(111)
            surf_tmp = ax1.imshow(np.flipud(J_mat),cmap=plt.cm.gray,extent=[self.crlb_x_grid[0],self.crlb_x_grid[-1],self.crlb_y_grid[0],self.crlb_y_grid[-1]])
            divider = make_axes_locatable(ax1)
            cax = divider.append_axes("right", size="5%", pad=0.05)            
            self.fig.colorbar(surf_tmp,cax=cax)
            save_name = os.path.join(self.LOG_DIR,'CRLB_costmat_'+str(self.log_state_ar.shape[0])+'.png')
            ax1.tick_params(labelsize=FONTSIZE)
            ax1.set_xlabel('X coordinate',fontsize=FONTSIZE)
            ax1.set_ylabel('Y coordinate',fontsize=FONTSIZE)
            self.fig.savefig(save_name,format='png',dpi=DPI,bbox_inches='tight')
            plt.close(self.fig)

            f_name = os.path.join(self.LOG_DIR,'CRLB_costmat_'+str(self.log_state_ar.shape[0])+'.h5')
            h5f = h5py.File(f_name, 'w')
            h5f.create_dataset('crlb_mat', data=J_mat)
            h5f.close()
        
        opt_move_idx = np.unravel_index(J_mat.flatten().argsort()[0],J_mat.shape)
        #print opt_move_idx
        opt_move = np.zeros((3,))        
        opt_move[0] = self.crlb_x_grid[opt_move_idx[1]]
        opt_move[1] = self.crlb_y_grid[opt_move_idx[0]]
        #opt_move = rotate_point(opt_move,self.log_state_ar[-1,2]*np.pi/180.)
        
        opt_phi = calc_orientation(self.log_state_ar[-1,:]+opt_move,Pos_Est)
        self.write_log_opt(opt_move,opt_phi)
        return J_mat,opt_move,opt_phi

            
        
    def validate_move(self,move,pos_est):
        new_pos = self.log_state_ar[-1,:]+move.reshape(3,)
        #print 'New_pos.size(): ', new_pos.shape

        if move[0] == 0 and move[1]==0:
            return False 
        
        for mm in range(self.N_mic):
            tmp_mic_pos = new_pos + self.mic_array_geo[mm,:]
            #print 'tmp_mic_pos.size(): ', tmp_mic_pos.shape
            #print tmp_mic_pos
            if tmp_mic_pos[0] <= 0 or tmp_mic_pos[0] >= self.room_size[0]:
                return False
            if tmp_mic_pos[1] <= 0 or tmp_mic_pos[1] >= self.room_size[1]:
                return False

        # This is to prevent a position, where two sources and robot are located on a single line, since this will make it impossible to distinguish for TDoA 
        N_sources = pos_est.shape[0]
        if N_sources > 1:
            angle_ar = np.zeros(N_sources,)
            for kk in range(N_sources):
                angle_ar[kk] = np.arctan2(pos_est[kk,0]-new_pos[0],pos_est[kk,1]-new_pos[1])*180./np.pi
                
            Dmat = np.tile(angle_ar,(N_sources,1))
            Diff_mat = np.abs(Dmat-Dmat.T)
            np.fill_diagonal(Diff_mat,170.)
            if np.min(Diff_mat) < 5.:
                return False
            
        return True



def nan_helper(y):
    """Helper to handle indices and logical indices of NaNs.

    Input:
        - y, 1d numpy array with possible NaNs
    Output:
        - nans, logical indices of NaNs
        - index, a function, with signature indices= index(logical_indices),
          to convert logical indices of NaNs to 'equivalent' indices
    Example:
        >>> # linear interpolation of NaNs
        >>> nans, x= nan_helper(y)
        >>> y[nans]= np.interp(x(nans), x(~nans), y[~nans])
    """

    return np.isnan(y), lambda z: z.nonzero()[0]


def main(args):
    rospy.init_node('sir_opt_pos')
    #print prev_Phi.shape
    ic = SirOptNav() 
    try:
        rospy.spin()
    except rospy.ROSInterruptException:
        pass

if __name__ == '__main__':
    main(sys.argv)
