#!/usr/bin/env python
import sys, time
import os
import numpy as np
import matplotlib.pyplot as plt
import scipy.spatial.distance as dist

LASER_RES = 180./509
SEG_LENGTH = 4
SEG_MEAN = 2
SEG_STD = 0.04
SEG_WIDTH_MIN = 0.05
SEG_WIDTH_MAX = 0.25


def find_segments(x,max_diff): #Finds the segments
	x = np.r_[0,x]
	x_diff = np.abs(np.diff(x))
	x_diff[x_diff<max_diff] = 0
	x_diff[x_diff>=max_diff] = max_diff
	segments = get_zero_runs(x_diff)
	return segments,x_diff

def get_zero_runs(a):
    # Create an array that is 1 where a is 0, and pad each end with an extra 0.
    iszero = np.concatenate(([0], np.equal(a, 0).view(np.int8), [0]))
    absdiff = np.abs(np.diff(iszero))
    # Runs start and end where absdiff is 1.
    ranges = np.where(absdiff == 1)[0].reshape(-1, 2)
    return ranges

def calc_coordinate(angle,r):
	angle = angle-90.
	x_coord = np.sin(angle*np.pi/180.)*r
	y_coord = np.cos(angle*np.pi/180.)*r
	return x_coord,y_coord

def calc_and_filter_segment_stats(segments,x):
	N_segments = len(segments)
	delete_idx = []
	segment_stats = []

	

	for ii in range(N_segments):
		tmp_center = (segments[ii,0]+(segments[ii,1]-segments[ii,0])/2.) * LASER_RES
		tmp_len = segments[ii,1]-segments[ii,0]
		tmp_mean = np.mean(x[segments[ii,0]:segments[ii,1]])
		tmp_std = np.std(x[segments[ii,0]:segments[ii,1]])
		tmp_angle_span = tmp_len*LASER_RES
		tmp_width = tmp_mean * np.sin(tmp_angle_span*np.pi/180.) / np.sin((180.-tmp_angle_span)/2.*np.pi/180.)
		tmp_x_coord,tmp_y_coord = calc_coordinate(tmp_center,tmp_mean) #Coordinates relative to the laser scanner (robot)
		if ii == 0:
			coord_ar = np.array([tmp_x_coord,tmp_y_coord])
		else:
			coord_ar = np.r_['0,2',coord_ar,[tmp_x_coord,tmp_y_coord]]
		if tmp_len > SEG_LENGTH and tmp_mean < SEG_MEAN and tmp_std < SEG_STD and tmp_width > SEG_WIDTH_MIN and tmp_width < SEG_WIDTH_MAX:
			segment_stats.append([tmp_center,tmp_len,tmp_mean,tmp_std,tmp_width,tmp_x_coord,tmp_y_coord])
		else:
			delete_idx.append(ii)

	if len(delete_idx) < N_segments:
		filtered_segments = np.delete(segments,delete_idx,axis=0)
		coord_ar = np.delete(coord_ar,delete_idx,axis=0)
	else:
		filtered_segments = []

	N_filtered_segments = len(filtered_segments)
	#print coord_ar
	print filtered_segments.shape
	if N_filtered_segments > 1:
		inter_dist = dist.cdist(coord_ar,coord_ar,'euclidean')
		inter_dist[inter_dist==0] = np.inf
		keep_idx = np.min(inter_dist,axis=1)<0.4
		#print inter_dist
		#print keep_idx
		leg_segments = filtered_segments[keep_idx,:]
		print leg_segments.shape
		leg_stats = [segment_stats[i] for i in np.where(keep_idx)[0]]
	else:
		leg_stats = segment_stats
		leg_segments = filtered_segments



	return leg_segments,leg_stats



if __name__=='__main__':
	np.set_printoptions(precision=2,suppress=True)
	range_data = np.loadtxt('/home/sociobot/catkin_ws/src/leg_detection/data/LEG_LASER_SCAN_17_02_17_09_02_30/log_laser.txt')
	single_scan = range_data[-1,:]
	segments,x_diff = find_segments(single_scan,0.075)
	filtered_segments,segment_stats = calc_and_filter_segment_stats(segments,single_scan)
	print '+++++++++++++++'
	print segments
	print segment_stats
	print type(filtered_segments),filtered_segments.shape
	plt.figure()
	plt.subplot(211)
	plt.plot(single_scan)
	plt.subplot(212)
	plt.plot(x_diff)
	plt.show()



